/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using Ascon.Pilot.DataClasses;
using Ascon.Pilot.Server.Api;
using Ascon.Pilot.Server.Api.Contracts;
using Ascon.Pilot.Transport;

namespace Pilot.Xps.Connection.Server
{
    interface IRemoteApi
    {
        IServerApi GetServerApi();
        IFileArchiveApi GetFileArchiveApi();
        IRemoteCommandsApi GetRemoteCommandsApi();
        IMessagesApi GetMessagesApi();
    }

    interface IRemoteApiListener
    {
        void NotifyMetadataChanged(long version);
        void NotifyChangeSet(DChangeset changeSet);
    }

    class RemoteApi : IRemoteApi, IServerCallback, IMessageCallback
    {
        private readonly IServerApi _serverApi;
        private readonly IRemoteCommandsApi _remoteCommandsApi;
        private readonly IFileArchiveApi _fileArchiveApi;
        private readonly IMessagesApi _messageApi;

        private IRemoteApiListener _remoteApiListener;

        public RemoteApi(HttpPilotClient client, string commandProcessorName)
        {
            _serverApi = client.GetServerApi(this);
            _fileArchiveApi = client.GetFileArchiveApi();
            _messageApi = client.GetMessagesApi(this);
            _remoteCommandsApi = new RemoteCommandsApi(client.GetServerCommandApi, commandProcessorName);
        }

        public void SetRemoteApiListener(IRemoteApiListener listener)
        {
            _remoteApiListener = listener;
        }

        public IServerApi GetServerApi()
        {
            return _serverApi;
        }

        public IFileArchiveApi GetFileArchiveApi()
        {
            return _fileArchiveApi;
        }

        public IRemoteCommandsApi GetRemoteCommandsApi()
        {
            return _remoteCommandsApi;
        }

        public IMessagesApi GetMessagesApi()
        {
            return _messageApi;
        }

        public void NotifyChangeset(DChangeset changeset)
        {
            if (_remoteApiListener == null || changeset == null)
                return;

            _remoteApiListener.NotifyChangeSet(changeset);
        }

        public void NotifyOrganisationUnitChangeset(OrganisationUnitChangeset changeset)
        {
        }

        public void NotifyPersonChangeset(PersonChangeset changeset)
        {
        }

        public void NotifyDMetadataChangeset(DMetadataChangeset changeset)
        {
            if (_remoteApiListener == null || changeset == null || changeset.Version == 0)
                return;

            _remoteApiListener.NotifyMetadataChanged(changeset.Version);
        }

        public void NotifySearchResult(DSearchResult searchResult)
        {
        }

        public void NotifyGeometrySearchResult(DGeometrySearchResult searchResult)
        {
        }

        public void NotifyDNotificationChangeset(DNotificationChangeset changeset)
        {
        }

        public void NotifyCommandResult(Guid requestId, byte[] data, ServerCommandResult result)
        {
        }

        public void NotifyMessageCreated(NotifiableDMessage message)
        {
        }

        public void NotifyTypingMessage(Guid chatId, int personId)
        {
        }

        public void CreateNotification(DNotification notification)
        {
        }

        public void NotifyOnline(int personId)
        {
        }

        public void NotifyOffline(int personId)
        {
        }

        public void UpdateLastMessageDate(DateTime maxDate)
        {
        }

        public void NotifyChangeAsyncCompleted(DChangeset changeset)
        {
        }

        public void NotifyChangeAsyncError(Guid identity, ProtoExceptionInfo exception)
        {
        }

        public void NotifyCustomNotification(string name, byte[] data)
        {
        }
    }
}
