/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.IO;
using System.Security.Cryptography.X509Certificates;
using Ascon.Pilot.Common.Connection;
using Ascon.Pilot.DataModifier;
using Ascon.Pilot.Server.Api;
using Pilot.Xps.Domain;
using Pilot.Xps.Domain.CertificateStorage;
using Pilot.Xps.Domain.Context;
using Pilot.Xps.Domain.Signatures;
using Pilot.Xps.Entities;

namespace Pilot.Xps.Connection.Server
{
    public class XpsServiceContext : IDisposable, IConnectionListener
    {
        private readonly IAppSettings _appSettings;
        private ConnectionCredentials _credentials;
        private HttpPilotClient _client;
        private ConnectionService _connectionService;
        private XpsServiceApi _xpsServiceApi;

        public XpsServiceContext(IAppSettings appSettings)
        {
            _appSettings = appSettings;
        }

        public ConnectionState GetConnectionState()
        {
            if (_connectionService == null)
                return ConnectionState.Offline;

            return _connectionService.IsConnected ? ConnectionState.Online : ConnectionState.Offline;
        }

        public void Start()
        {
            if (GetConnectionState() == ConnectionState.Online)
                return;

            Dispose();

            _credentials = ConnectionCredentials.GetConnectionCredentials(_appSettings.ConnectionSettings.ConnectionParams);
            _client = new HttpPilotClient(_credentials.GetConnectionString(), _credentials.GetConnectionProxy());
            _connectionService = new ConnectionService(_client, _credentials, this, _appSettings.ConnectionSettings);
            _connectionService.FirstConnect();
        }

        public void Stop()
        {
            if (GetConnectionState() != ConnectionState.Online)
                return;

            _connectionService.Disconnect();
        }
        
        public void Connected()
        {
            var remoteApi = new RemoteApi(_client, XpsServerConstants.XpsServiceName);
            var commandsApi = remoteApi.GetRemoteCommandsApi();
            var archiveApi = remoteApi.GetFileArchiveApi();
            var serverApi = remoteApi.GetServerApi();
            var messageApi = remoteApi.GetMessagesApi();
            var databaseInfo = serverApi.OpenDatabase();

            var fileStorageProvider = new FileSystemStorageProvider(GetFileStorageFolder());
            var fileLoader = new FileLoader(archiveApi);
            var xpsMergeService = new XpsDocumentMergeService(fileLoader);
            var changeSetUploader = new ChangesetUploader(archiveApi, fileStorageProvider);
            var backend = new Backend(serverApi, messageApi, changeSetUploader);
            var documentSigner = new XpsDocumentSigner(backend, fileLoader, fileStorageProvider);
            var certificateStorage = new CertificateStorage(_appSettings.SigningSettings.ToCertificateStorageSettings());
            _xpsServiceApi = new XpsServiceApi(xpsMergeService, documentSigner, backend, certificateStorage);

            commandsApi.RegisterCommandHandler<IXpsServiceApi>(_xpsServiceApi);
        }

        public void Disconnected(Exception exception = null)
        {
            Dispose();
        }

        public void Dispose()
        {
            _client?.Dispose();
        }

        private string GetFileStorageFolder()
        {
            var cod = Path.Combine(Environment.GetFolderPath(Environment.SpecialFolder.CommonApplicationData), "ASCON");
            return Path.Combine(cod, "Pilot-XPS-Service", "FileStorage");
        }
    }

    static class CertificateStorageSettingsEx
    {
        public static ICertificateStorageSettings ToCertificateStorageSettings(this ISigningSettings settings)
        {
            return new CertificateStorageSettings(settings);
        }
    }

    class CertificateStorageSettings : ICertificateStorageSettings
    {
        public CertificateStorageSettings(ISigningSettings settings)
        {
            StoreLocation = settings.StoreLocation;
            CertificatesMap = settings.CertificatesMap;
        }

        public StoreLocation StoreLocation { get; set; }
        public IDictionary<string, string> CertificatesMap { get; set; }
    }
}
