/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Linq;
using System.Reflection;
using System.ServiceProcess;
using System.Text;
using Ascon.Pilot.Common.DataProtection;
using Pilot.Xps.Connection.Server;

namespace Pilot.Xps.Service
{
    class Program
    {
        public static string ServiceName = "Pilot.Xps.Service";
        private static XpsServiceContext _context;
        private static XpsServiceContextLogger _contextLogger;

        static void Main(string[] args)
        {
            if (!Environment.UserInteractive)
                // running as service
                using (var service = new Service())
                    ServiceBase.Run(service);
            else
            {
                if (args.Length == 0)
                {
                    Console.WriteLine($"Type {ServiceName} --help, -h, -?' for help.");
                    return;
                }

                if (args.Contains("-h") || args.Contains("--help") || args.Contains("?"))
                {
                    PrintHelp(args);
                    return;
                }

                if (args.Contains("-v") || args.Contains("--version"))
                {
                    var version = Assembly.GetExecutingAssembly().GetName().Version.ToString();
                    Console.WriteLine(version);
                    return;
                }

                if (args.Contains("-s") || args.Contains("--status"))
                {
                    var settings = AppSettingsProvider.LoadAppSettings();
                    Console.WriteLine("Connection parameters:");
                    Console.WriteLine($"    Server url {settings.ConnectionSettings.ConnectionParams.Server}");
                    Console.WriteLine($"    Username {settings.ConnectionSettings.ConnectionParams.UserName}");
                    Console.WriteLine($"    License code {settings.ConnectionSettings.LicenseCode}");
                    return;
                }

                if (args.Contains("-i") || args.Contains("--install"))
                {
                    var filename = Assembly.GetExecutingAssembly().Location;
                    ServiceInstaller.InstallAndStart(ServiceName, ServiceName, filename);
                    Console.WriteLine("Service installed successfully");
                    return;
                }

                if (args.Contains("-u") || args.Contains("--uninstall"))
                {
                    ServiceInstaller.Uninstall(ServiceName);
                    Console.WriteLine("Service uninstalled successfully");
                    return;
                }

                if (args.Contains("-c") || args.Contains("--connect"))
                {
                    Connect(args);
                }

                // running as console app
                Console.WriteLine("Connecting...");
                Console.WriteLine(Start(args) ? "Connected" : "Not connected");
                Console.WriteLine("Press any key to stop...");
                Console.ReadKey(true);
                Stop();
            }
        }

        public static bool Start(string[] args)
        {
            var settings = AppSettingsProvider.LoadAppSettings();
            _context = new XpsServiceContext(settings);
            _contextLogger = new XpsServiceContextLogger(_context);
            return _contextLogger.Start();
        }

        public static void Stop()
        {
            _context.Dispose();
        }

        private static void Connect(string[] args)
        {
            if (args.Length > 1)
            {
                var databaseUrl = args[1];

                Console.Write("login: ");
                var login = System.Console.ReadLine();

                Console.Write("password: ");
                var password = GetPassword();

                var settings = AppSettingsProvider.LoadAppSettings();
                settings.ConnectionSettings.ConnectionParams.Server = databaseUrl;
                settings.ConnectionSettings.ConnectionParams.UserName = login;
                settings.ConnectionSettings.ConnectionParams.Password = password.ConvertToSecureString();
                AppSettingsProvider.SaveAppSettings(settings);
            }

            Console.WriteLine();
        }

        private static string GetPassword()
        {
            var password = "";
            ConsoleKeyInfo key;
            do
            {
                key = Console.ReadKey(true);
                if (!char.IsControl(key.KeyChar))
                {
                    password += key.KeyChar;
                    Console.Write("*");
                }
                else
                {
                    if (key.Key == ConsoleKey.Backspace && password.Length > 0)
                    {
                        password = password.Remove(password.Length - 1, 1);
                        Console.Write("\b \b");
                    }
                }
            }

            while (key.Key != ConsoleKey.Enter);
            Console.WriteLine();
            return password;
        }

        private static void PrintHelp(string[] args)
        {
            Console.WriteLine($"{ServiceName} command-line client.");
            Console.WriteLine();
            Console.WriteLine($"Usage: {ServiceName} <command> [args]");
            Console.WriteLine("Available commands:");

            var strBuilder = new StringBuilder();
            strBuilder.AppendLine($"-v, --version   Show {ServiceName} version");
            strBuilder.AppendLine($"-i, --install   Install {ServiceName} as windows service");
            strBuilder.AppendLine("-u, --uninstall    Removing a service entry from the registry");
            strBuilder.AppendLine("-c, --connect {database url with port}   Connect to the Pilot-Server with new parameters");
            strBuilder.AppendLine("     for example:    --connect http://localhost:5545/base");
            Console.WriteLine(strBuilder.ToString());
        }
    }
}
