/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using Ascon.Pilot.DataClasses;
using PilotRocketChatGateway.Utils;
using System;
using System.ServiceModel.Channels;

namespace PilotRocketChatGateway.UserContext
{
    public interface IDataLoader
    {
        IRCDataConverter RCDataConverter { get; }
        Room LoadRoom(DChat chat, DMessage lastMessage);
        IList<Room> LoadRooms(string updatedSince);
        Subscription LoadRoomsSubscription(DChatInfo chat);
        IList<Subscription> LoadRoomsSubscriptions(string updatedSince);
        (IList<User>, int) LoadUsers(int count, int offset, string text);
        (IList<Room>, int) LoadChannels(int count, int offset, string text);
        Room LoadPersonalRoom(string username);
        IList<Message> LoadMessages(string roomId, int count, string upperBound);
        IList<Message> LoadMessages(string roomId, int count, int offset, long upperBoundInMilliseconds);
        IList<Message> LoadMessages(string roomId, string lowerBound);
        IList<Message> LoadPreviousMessages(string roomId, int count, int offset, long lowerBoundInMilliseconds);
        Message LoadMessage(string msgId);
        IList<Message> LoadSurroundingMessages(string rcMsgId, string roomId, int count);
        User LoadUser(int usderId);
        (IList<User>, int) LoadMembers(string roomId,int count, int offset, string text);
        DChatInfo LoadChat(Guid chatId);
        INPerson LoadPerson(int userId);
        (IEnumerable<Message>, long) SearchMessages(string roomId, string searchText, int count, int offset);
    }
    public class DataLoader : IDataLoader
    {
        private readonly ICommonDataConverter _commonConverter;
        private readonly IContext _context;
        private readonly IBatchMessageLoader _loader;
        private readonly ILogger _logger;
        private Guid _previousSearchId = Guid.Empty;

        public DataLoader(IRCDataConverter rcConverter, ICommonDataConverter commonConverter, IContext context, IBatchMessageLoader msgLoader, ILogger logger)
        {
            RCDataConverter = rcConverter;
            _commonConverter = commonConverter;
            _context = context;
            _loader = msgLoader;
            _logger = logger;
        }
        public IRCDataConverter RCDataConverter { get; }
        public Room LoadRoom(DChat chat, DMessage lastMessage)
        {
            return RCDataConverter.ConvertToRoom(chat, lastMessage);
        }
        public DChatInfo LoadChat(Guid chatId)
        {
            return _context.RemoteService.ServerApi.GetChat(chatId);
        }

        public IList<Room> LoadRooms(string updatedSince)
        {
            DateTime lastUpdated = GetChatListLastUpdated(updatedSince);
            var chats = _context.RemoteService.ServerApi.GetChats(lastUpdated);
            var result = new List<Room>();

            foreach (var chat in chats)
            {
                try
                {
                    var rcChat = RCDataConverter.ConvertToRoom(chat.Chat, chat.LastMessage);
                    result.Add(rcChat);
                }
                catch (Exception e)
                {
                    _logger.LogError(e.Message);
                }
            }
            return result;
        }

        public (IList<Room>, int) LoadChannels(int count, int offset, string text)
        {
            var result = new List<Room>();
            var allChats = _context.RemoteService.ServerApi.GetChats(DateTime.MinValue).Where(x => x.Chat.Type != ChatKind.Personal).OrderByDescending(x => x.LastMessage.ServerDate).ToList();
            var filtered = FilterChats(allChats, text);

            foreach (var chat in filtered.Skip(offset).Take(count))
            {
                try
                {
                    var rcChat = RCDataConverter.ConvertToRoom(chat.Chat, chat.LastMessage);
                    var members = _context.RemoteService.ServerApi.GetChatMembers(chat.Chat.Id);
                    rcChat.usersCount = members.Count;
                    result.Add(rcChat);
                }
                catch (Exception e)
                {
                    _logger.LogError(e.Message);
                }
            }
            return (result, filtered.Count);
        }

        private DateTime GetChatListLastUpdated(string updatedSince)
        {
            var date = string.IsNullOrEmpty(updatedSince) ? DateTime.MinValue : _commonConverter.ConvertFromJSDate(updatedSince);

            if (date == DateTime.MinValue)
                return date;

            return date.AddDays(-1);
        }

        public Subscription LoadRoomsSubscription(DChatInfo chat)
        {
            return RCDataConverter.ConvertToSubscription(chat);
        }
        public IList<Subscription> LoadRoomsSubscriptions(string updatedSince)
        {
            var lastUpdated = GetChatListLastUpdated(updatedSince);
            var chats = _context.RemoteService.ServerApi.GetChats(lastUpdated);
            var result = new List<Subscription>();
            foreach (var chat in chats)
            {
                try
                {
                    var rcChat = RCDataConverter.ConvertToSubscription(chat);
                    result.Add(rcChat);
                }
                catch (Exception e)
                {
                    _logger.LogError(e.Message);
                }
            }
            return result;
        }
        public Room LoadPersonalRoom(string username)
        {
            var person = _context.RemoteService.ServerApi.GetPerson(username);
            var chat = _context.RemoteService.ServerApi.GetPersonalChat(person.Id);
            return chat.Chat.Id == Guid.Empty ? null : RCDataConverter.ConvertToRoom(chat.Chat, chat.LastMessage);
        }
        public IList<Message> LoadMessages(string roomId, int count, string upperBound)
        {
            var id = _commonConverter.ConvertToChatId(roomId);
            var dateTo = _commonConverter.ConvertFromJSDate(upperBound);
            return LoadMessages(id, DateTime.MinValue.ToUniversalTime(), dateTo.AddMilliseconds(-1), count);
        }
        public IList<Message> LoadMessages(string roomId, string lowerBound)
        {
            var id = _commonConverter.ConvertToChatId(roomId);
            var dateFrom = _commonConverter.ConvertFromJSDate(lowerBound);
            return LoadMessages(id, dateFrom, DateTime.MaxValue.ToUniversalTime(), int.MaxValue);
        }

        public IList<Message> LoadMessages(string roomId, int count, int offset, long upperBoundInMilliseconds)
        {
            var id = _commonConverter.ConvertToChatId(roomId);
            var dateTo = _commonConverter.ConvertFromJSDate(upperBoundInMilliseconds);
            return LoadMessages(id, DateTime.MinValue.ToUniversalTime(), dateTo.AddMilliseconds(-1), int.MaxValue).OrderByDescending(x => x.ServerDate).Skip(offset).Take(count).ToList();
        }

        public IList<Message> LoadPreviousMessages(string roomId, int count, int offset, long lowerBoundInMilliseconds)
        {
            var id = _commonConverter.ConvertToChatId(roomId);
            var dateFrom = _commonConverter.ConvertFromJSDate(lowerBoundInMilliseconds);
            return LoadMessages(id, dateFrom, DateTime.MaxValue.ToUniversalTime(), int.MaxValue).Take(count).OrderBy(x => x.ServerDate).ToList();
        }

        public Message LoadMessage(string msgId)
        {
            DMessage? msg = _commonConverter.IsRocketChatId(msgId) ?
                _context.RemoteService.ServerApi.GetMessage(msgId) :
                _context.RemoteService.ServerApi.GetMessage(Guid.Parse(msgId));

            var chat = _context.RemoteService.ServerApi.GetChat(msg.ChatId);
            return RCDataConverter.ConvertToMessage(msg, chat.Chat);
        }

        public IList<Message> LoadSurroundingMessages(string rcMsgId, string roomId, int count)
        {
            Guid msgId = _commonConverter.ConvertToMsgId(rcMsgId);
            Guid chatId = _commonConverter.ConvertToChatId(roomId);
            var chat = _context.RemoteService.ServerApi.GetChat(chatId);
            var messages = _loader.FindMessage(msgId, chatId, count);
            return messages.Select(x => RCDataConverter.ConvertToMessage(x, chat.Chat)).ToList();
        }

        public User LoadUser(int userId)
        {
            var person = _context.RemoteService.ServerApi.GetPerson(userId);
            return _commonConverter.ConvertToUser(person);
        }
        public INPerson LoadPerson(int userId)
        {
            return _context.RemoteService.ServerApi.GetPerson(userId);
        }
        public (IList<User>, int) LoadUsers(int count, int offset, string text)
        {
            var result = new List<User>();
            var allUsers = _context.RemoteService.ServerApi.GetPeople().Values.Where(x => !x.IsDeleted && x.Login != _context.UserData.Username).ToList();
            var filtered = FilterUsers(allUsers, text);

            foreach(var user in filtered.Skip(offset).Take(count))
            {
                try
                {
                    var rcUser = _commonConverter.ConvertToUser(user);
                    result.Add(rcUser);
                } catch (Exception e)
                {
                    _logger.LogError(e.Message);
                }
            }
            return (result, filtered.Count());
        }

        private List<INPerson> FilterUsers(List<INPerson> allUsers, string text)
        {
            if (string.IsNullOrEmpty(text))
                return allUsers;

            return allUsers.Where(x => x.Login.ToLower().Contains(text) || x.DisplayName.ToLower().Contains(text)).ToList();
        }

        private List<DChatMember> FilterMembers(List<DChatMember> allMembers, string text)
        {
            if (string.IsNullOrEmpty(text))
                return allMembers;

            var result = new List<DChatMember>();
            foreach (var member in allMembers)
            {
                var person = _context.RemoteService.ServerApi.GetPerson(member.PersonId);
                if (person.Login.ToLower().Contains(text) || person.DisplayName.ToLower().Contains(text))
                    result.Add(member);
            }

            return result;
        }

        private List<DChatInfo> FilterChats(List<DChatInfo> allChats, string text)
        {
            if (string.IsNullOrEmpty(text))
                return allChats;

            return allChats.Where(x => x.Chat.Name.ToLower().Contains(text)).ToList();
        }

        public (IList<User>, int) LoadMembers(string roomId, int count, int offset, string text)
        {
            var result = new List<User>();
            var id = _commonConverter.ConvertToChatId(roomId);
            var members = _context.RemoteService.ServerApi.GetChatMembers(id);
            var filtered = FilterMembers(members, text);

            foreach (var member in filtered.Skip(offset).Take(count))
            {
                try
                {
                    var person = _context.RemoteService.ServerApi.GetPerson(member.PersonId);
                    var rcUser = _commonConverter.ConvertToUser(person);
                    result.Add(rcUser);
                }
                catch (Exception e)
                {
                    _logger.LogError(e.Message);
                }
            }
            return (result, filtered.Count);
        }

        public bool IsChatNotifiable(Guid chatId)
        {
            var member = _context.RemoteService.ServerApi.GetChatMembers(chatId).First(x => x.PersonId == _context.RemoteService.ServerApi.CurrentPerson.Id);
            return member.IsNotifiable;
        }

        private IList<Message> LoadMessages(Guid roomId, DateTime dateFrom, DateTime dateTo, int count)
        {
            var msgs = _context.RemoteService.ServerApi.GetMessages(roomId, dateFrom, dateTo, count);

            if (msgs == null)
                return new List<Message>();

            var chat = _context.RemoteService.ServerApi.GetChat(roomId);
            var result = new List<Message>();
            foreach (var msg in msgs)
            {
                try
                {
                    var rcMsg = RCDataConverter.ConvertToMessage(msg, chat.Chat);
                    result.Add(rcMsg);
                }
                catch (Exception e)
                {
                    _logger.LogError(e.Message);
                }
               
            }

            return result;
        }

        public (IEnumerable<Message>, long) SearchMessages(string roomId, string searchText, int count, int offset)
        {
            var terms = searchText.Trim().Split(" ");
            var searchString = ("+" + string.Join(" +", terms));

            var chatId = _commonConverter.ConvertToChatId(roomId);
            var searchId = CreateSearchId(offset);
            var (found, totalHits) = _context.RemoteService.ServerApi.SearchMessages(searchId, count, searchString, new List<Guid> { chatId });
            if (!found.Any())
                return (new List<Message>(), 0);

            var result = new List<Message>();
            var chat = _context.RemoteService.ServerApi.GetChat(chatId);
            foreach (var msgId in found)
            {
                var msg = _context.RemoteService.ServerApi.GetMessage(msgId);
                var rcMsg = RCDataConverter.ConvertToMessage(msg, chat.Chat);
                result.Add(rcMsg);
            }
            return (result, totalHits);
        }

        private Guid CreateSearchId(int offset)
        {
            var id = offset > 0 ? _previousSearchId : Guid.NewGuid();
            _previousSearchId = id;
            return id;
        }
    }
}
