/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System.Windows.Input;
using System.Windows.Media;
using Ascon.Pilot.Theme.Controls;
using Microsoft.Practices.Prism.Commands;
using PilotShare.Client.Utils;

namespace PilotShare.Client.ToolbarItems
{
    public interface IToolbarButtonItemBuilder
    {
        IToolbarButtonItemBuilder WithHeader(object header);
        IToolbarButtonItemBuilder WithIcon(ImageSource icon);
        IToolbarButtonItemBuilder WithCommand(ICommand command);
        IToolbarButtonItemBuilder WithCommandParameter(object commandParameter);
        IToolbarButtonItemBuilder WithIsEnabled(bool value);
        IToolbarButtonItemBuilder WithShowCaption(bool value);
        IToolbarButtonItemBuilder WithHint(string hint);
    }

    public class ToolbarButtonItem : PropertyChangedBase, IToolbarButtonItem, IToolbarButtonItemBuilder
    {
        private string _name;
        private object _header;
        private ImageSource _icon;
        private ICommand _command;
        private object _commandParameter;
        private bool? _isEnabled = true;
        private bool _showCaption = true;

        public ToolbarButtonItem(string name)
        {
            Name = name;
        }

        public object Header
        {
            get => _header;
            set
            {
                _header = value;
                NotifyOfPropertyChange(nameof(Header));
            }
        }

        public ImageSource Icon
        {
            get => _icon;
            set
            {
                _icon = value;
                NotifyOfPropertyChange(nameof(Icon));
            }
        }

        public ICommand Command
        {
            get => _command;
            set
            {
                _command = value;
                NotifyOfPropertyChange(nameof(Command));
            }
        }

        public object CommandParameter
        {
            get => _commandParameter;
            set
            {
                _commandParameter = value;
                NotifyOfPropertyChange(nameof(CommandParameter));

                if (Command is DelegateCommand canExecute)
                    canExecute.RaiseCanExecuteChanged();
            }
        }

        public bool? IsEnabled
        {
            get => _isEnabled;
            protected set
            {
                _isEnabled = value;
                NotifyOfPropertyChange(nameof(IsEnabled));
            }
        }

        public bool ShowCaption
        {
            get => _showCaption;
            set
            {
                _showCaption = value;
                NotifyOfPropertyChange(nameof(ShowCaption));
            }
        }

        public string Name
        {
            get => _name;
            set
            {
                _name = value;
                NotifyOfPropertyChange(nameof(Name));
            }
        }

        public IToolbarButtonItemBuilder WithCommand(ICommand command)
        {
            Command = command;
            return this;
        }
        public IToolbarButtonItemBuilder WithCommandParameter(object commandParameter)
        {
            CommandParameter = commandParameter;
            return this;
        }

        public IToolbarButtonItemBuilder WithHeader(object header)
        {
            Header = header;
            return this;
        }

        public IToolbarButtonItemBuilder WithIcon(ImageSource icon)
        {
            Icon = icon;
            return this;
        }

        public IToolbarButtonItemBuilder WithIsEnabled(bool value)
        {
            IsEnabled = value;
            return this;
        }

        public IToolbarButtonItemBuilder WithHint(string hint)
        {
            var command = Command as DelegateCommandBase;
            if (command != null && hint != command.Hint)
            {
                command.Hint = hint;
            }
            return this;
        }

        public IToolbarButtonItemBuilder WithShowCaption(bool value)
        {
            ShowCaption = value;
            return this;
        }
    }
}
