/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Input;
using Microsoft.Practices.Prism.Commands;
using PilotShare.Client.Utils;

namespace PilotShare.Client.View
{
    class EditLinkViewModel : PropertyChangedBase
    {
        protected readonly LinkItem _link;
        protected readonly ILinkObjectModifier _modifier;
        protected readonly IPasswordHolder _passwordHolder;

        private DelegateCommand _copyCommand;
        private DelegateCommand _submitCommand;
        private string _linkString;
        private DateTime? _expirationDate;

        public EditLinkViewModel(LinkItem link, ILinkObjectModifier modifier, IPasswordHolder passwordHolder)
        {
            _link = link;
            _modifier = modifier;
            _passwordHolder = passwordHolder;
            InitCommands();
            Init(link);
        }

        public ICommand CopyCommand => _copyCommand;
        public ICommand SubmitCommand => _submitCommand;

        public string LinkString
        {
            get => _linkString;
            set
            {
                _linkString = value;
                NotifyOfPropertyChange(nameof(LinkString));
            }
        }

        public DateTime? ExpirationDate
        {
            get => _expirationDate;
            set
            {
                _expirationDate = value;
                NotifyOfPropertyChange(nameof(ExpirationDate));
                Validate();
            }
        }

        public void Validate()
        {
            _submitCommand.RaiseCanExecuteChanged();
        }

        private void Init(LinkItem link)
        {
            LinkString = link.Url;
            ExpirationDate = link.ValidThrough;
            _passwordHolder.Password = link.Password;
        }

        private void InitCommands()
        {
            _copyCommand = new DelegateCommand(CopyToClipboard);
            _submitCommand = new DelegateCommand(Submit, CanSubmit);
        }

        protected virtual bool CanSubmit()
        {
            var currentPassword = _link.Password ?? string.Empty;
            return ExpirationDate != _link.ValidThrough || _passwordHolder.Password != currentPassword;
        }

        protected virtual void Submit()
        {
            _link.Password = _passwordHolder.Password;
            _link.ValidThrough = ExpirationDate;

            _modifier.ModifyLink(_link);
            CloseView(true);
        }

        private void CopyToClipboard()
        {
            Clipboard.SetText(LinkString);
        }
    }
}
