/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Threading;
using Ascon.Pilot.DataClasses;
using Ascon.Pilot.Server.Api;
using Ascon.Pilot.Server.Api.Contracts;
using Ascon.Pilot.Transport;

namespace PilotShare.Server.Models
{
    public class ServerApiWrapper : IServerApi, IFileArchiveApi, IDisposable
    {
        private int TIMEOUT = 10000;
        private readonly ConnectionCredentials _credentials;
        private readonly HttpPilotClient _client;
        private readonly int _licenseCode;
        private readonly IServerApi _serverApi;
        private readonly IAuthenticationApi _authApi;
        private readonly IFileArchiveApi _fileArchiveApi;
        private bool _isConnected;

        public ServerApiWrapper(ConnectionCredentials credentials, HttpPilotClient client, int licenseCode)
        {
            _credentials = credentials;
            _client = client;
            _licenseCode = licenseCode;
            _authApi = client.GetAuthenticationApi();
            _serverApi = client.GetServerApi(new NullableServerCallback());
            _fileArchiveApi = client.GetFileArchiveApi();
        }

        private TResponse Request<TResponse>(Func<TResponse> func)
        {
            Stopwatch sw = Stopwatch.StartNew();
            bool success = false;
            while (!success)
            {
                if (sw.ElapsedMilliseconds > TIMEOUT)
                    throw new TimeoutException();

                try
                {
                    if (!_isConnected)
                        Connect();
                    success = true;
                }
                catch (TransportException)
                {
                    _isConnected = false;
                    Thread.Sleep(10);
                }
            }
            return func();
        }

        public void Connect()
        {
            _client.Connect(false);
            _authApi.Login(_credentials.DatabaseName, _credentials.Username, _credentials.ProtectedPassword, _credentials.UseWindowsAuth, _licenseCode);
            _serverApi.OpenDatabase();
            _isConnected = true;
        }

        public byte[] GetFileChunk(Guid id, long pos, int count)
        {
            return Request(() => _fileArchiveApi.GetFileChunk(id, pos, count));
        }

        public List<DObject> GetObjects(Guid[] ids)
        {
            return Request(() => _serverApi.GetObjects(ids));
        }

        public DDatabaseInfo OpenDatabase()
        {
            return _serverApi.OpenDatabase();
        }

        public DDatabaseInfo GetDatabase(string database)
        {
            return _serverApi.GetDatabase(database);
        }

        public DDatabaseInfo GetDatabaseInfo()
        {
            return _serverApi.GetDatabaseInfo();
        }

        public DMetadata GetMetadata(long localVersion)
        {
            return _serverApi.GetMetadata(localVersion);
        }

        public DSettings GetPersonalSettings()
        {
            return _serverApi.GetPersonalSettings();
        }

        public DSettings GetCommonSettings()
        {
            return _serverApi.GetCommonSettings();
        }

        public void ChangeSettings(DSettingsChange change)
        {
            _serverApi.ChangeSettings(change);
        }

        public List<DChangeset> GetChangesets(long first, long last)
        {
            return _serverApi.GetChangesets(first, last);
        }

        public DChangeset Change(DChangesetData changes)
        {
            return _serverApi.Change(changes);
        }

        public void ChangeAsync(DChangesetData changesetData)
        {
            _serverApi.ChangeAsync(changesetData);
        }

        public void PutFileChunk(Guid id, byte[] buffer, long pos)
        {
            _fileArchiveApi.PutFileChunk(id, buffer, pos);
        }


        public long GetFilePosition(Guid id)
        {
            return _fileArchiveApi.GetFilePosition(id);
        }

        public void PutFileInArchive(DFileBody fileBody)
        {
            _fileArchiveApi.PutFileInArchive(fileBody);
        }

        public List<DPerson> LoadPeople()
        {
            return _serverApi.LoadPeople();
        }

        public List<DPerson> LoadPeopleByIds(int[] ids)
        {
            return _serverApi.LoadPeopleByIds(ids);
        }

        public List<DOrganisationUnit> LoadOrganisationUnits()
        {
            return _serverApi.LoadOrganisationUnits();
        }

        public List<DOrganisationUnit> LoadOrganisationUnitsByIds(int[] ids)
        {
            return _serverApi.LoadOrganisationUnitsByIds(ids);
        }

        public void AddSearch(DSearchDefinition searchDefinition)
        {
            _serverApi.AddSearch(searchDefinition);
        }

        public void RemoveSearch(Guid searchDefinitionId)
        {
            _serverApi.RemoveSearch(searchDefinitionId);
        }

        public void UpdatePerson(DPersonUpdateInfo updateInfo)
        {
            _serverApi.UpdatePerson(updateInfo);
        }

        public void UpdateOrganisationUnit(DOrganisationUnitUpdateInfo updateInfo)
        {
            _serverApi.UpdateOrganisationUnit(updateInfo);
        }

        public IEnumerable<DHistoryItem> GetHistoryItems(Guid[] ids)
        {
            return _serverApi.GetHistoryItems(ids);
        }

        public void InvokeCommand(string commandName, Guid requestId, byte[] data)
        {
            _serverApi.InvokeCommand(commandName, requestId, data);
        }

        public void SubscribeCustomNotification(string name)
        {
            _serverApi.SubscribeCustomNotification(name);
        }

        public void UnsubscribeCustomNotification(string name)
        {
            _serverApi.UnsubscribeCustomNotification(name);
        }

        public void SendCustomNotification(string name, byte[] data)
        {
            _serverApi.SendCustomNotification(name, data);
        }

        public IEnumerable<AccessRecord> LoadAccessRecords(Guid objectId)
        {
            return _serverApi.LoadAccessRecords(objectId);
        }

        public void Dispose()
        {
            _client.Dispose();
        }
    }
}
