/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.IO;
using System.Net;
using System.Reflection;
using Microsoft.Extensions.Options;

namespace PilotShare.Server.Models
{
    public interface IServerVersionInfoService
    {
        string PilotServerVersion { get; }
        string PilotShareServerVersion { get; }
    }

    public class ServerVersionInfoService : IServerVersionInfoService
    {
        private readonly AuthSettings _authSettings;
        public ServerVersionInfoService(IOptions<AuthSettings> config)
        {
            _authSettings = config.Value;
            var response = GetPilotServerVersion();
            PilotServerVersion = response;
            PilotShareServerVersion = GetAssemblyVersion();
        }

        public string PilotServerVersion { get; }
        public string PilotShareServerVersion { get; }

        private string GetBaseServerUrl(string fullUrl)
        {
            var uri =  new Uri(fullUrl);
            var url = $"{uri.Scheme}://{uri.Host}:{uri.Port}";
            return url;
        }

        private string GetPilotServerVersion()
        {
            var url = GetBaseServerUrl(_authSettings.Url);
            var response = WebRequestHelper.Request(url);
            var split = response.Substring(response.IndexOf("_v", StringComparison.InvariantCultureIgnoreCase) + 2);
            return split;
        }

        private string GetAssemblyVersion()
        {
            var version = (Assembly.GetEntryAssembly() ?? Assembly.GetCallingAssembly()).GetName().Version;
            return version?.ToString();
        }
    }

    class WebRequestHelper
    {
        public static string Request(string url)
        {
            // Create a request for the URL.
            WebRequest request = WebRequest.Create(url);
            // If required by the server, set the credentials.
            request.Credentials = CredentialCache.DefaultCredentials;

            // Get the response.
            WebResponse response = request.GetResponse();
            // Display the status.
            Console.WriteLine(((HttpWebResponse)response).StatusDescription);

            // Get the stream containing content returned by the server.
            // The using block ensures the stream is automatically closed.
            using (Stream dataStream = response.GetResponseStream())
            {
                // Open the stream using a StreamReader for easy access.
                StreamReader reader = new StreamReader(dataStream);
                // Read the content.
                string responseFromServer = reader.ReadToEnd();

                // Close the response.
                response.Close();

                // Display the content.
                return responseFromServer;
            }
        }
    }
}
