﻿using Ascon.Pilot.Common;
using Ascon.Pilot.DataClasses;
using Ascon.Pilot.Server.Api;
using Ascon.Pilot.Server.Api.Contracts;

namespace UrlsHandlerSample.Services
{
    public interface IPilotObjectInfoService
    {
        Task<DObject> GetObjectInfo(Guid id);
    }

    public interface IPilotServiceConnector
    {
        Task Connect(string serverUrl, string databaseName, string login, string password, int license);
        HttpPilotClient? GetClient();
    }

    internal class PilotObjectInfoService : IPilotObjectInfoService, IPilotServiceConnector, IDisposable
    {
        private HttpPilotClient? _pilotClient;
        private IServerApi? _serverApi;

        public Task<DObject> GetObjectInfo(Guid id)
        {
            if (_pilotClient == null || _serverApi == null)
                throw new Exception("The connection to the Pilot-Server is not established. Please connect to the Pilot-Server first.");

            return Task.Factory.StartNew(() => 
            {
                var dObject = _serverApi.GetObjects(new [] {id});
                return dObject.First();
            });
        }

        public async Task Connect(string serverUrl, string databaseName, string login, string password, int license)
        {
            _pilotClient = new HttpPilotClient(serverUrl);
            _pilotClient.Connect(false);
            var authApi = _pilotClient.GetAuthenticationAsyncApi();
            await authApi.LoginAsync(databaseName, login, password.EncryptAes(), license);

            _serverApi = _pilotClient.GetServerApi(null);
            _serverApi.OpenDatabase();
        }

        public void Dispose()
        {
            _pilotClient?.Dispose();
        }

        public HttpPilotClient? GetClient()
        {
            return _pilotClient;
        }
    }
}
