/*
Copyright © 2024 ASCON-Design Systems LLC. All rights reserved.
This sample is licensed under the MIT License.
*/
import { CryptoProAdapter } from "./cryptopro/crypto-pro.adapter";
import {
  ICertificate,
  ICryptoProvider,
  IDisposable,
  IInitializable,
  InjectionSource,
  ISignatureVerificationResult,
  ISignatureRequest,
  IFile,
  IImportedSignatureVerificationResult
} from "@pilotdev/pilot-web-sdk";
import { Observable } from "rxjs";
import { Queue, QueueType } from "./utils/queue";
import { FileSignatureUpdater } from "./file-signature.updater";
import { CryptoProExtensionModel } from "./cryptoProExtensionModel";
import { CertificateValidator } from "./cryptopro/certificate.validator";
import { AdvancementSettingsProvider } from "./advancement/advancementSettingsProvider";

export class CryptoProExtension implements ICryptoProvider, IInitializable, IDisposable {
  private _model!: CryptoProExtensionModel;
  
  initialize(injectionSource: InjectionSource): void {
    
    const personalSettings = new AdvancementSettingsProvider(injectionSource.personalSettings);
    const fileSignatureUpdater = new FileSignatureUpdater(injectionSource.modifierProvider, personalSettings);
    const certificateValidator = new CertificateValidator();
    const cryptoProAdapter = new CryptoProAdapter(certificateValidator, (error) => this._model.onCryptoProPluginLoaded(error), injectionSource.objectsRepository);
    const queue = new Queue<QueueType>();

    this._model = new CryptoProExtensionModel(cryptoProAdapter, fileSignatureUpdater, queue, personalSettings);
    this._model.initialize();
  }

  dispose(): void {
    this._model?.dispose();
  }

  sign(documentId: string, actualFile: IFile, arrayBuffer: ArrayBuffer, certificate: ICertificate, signatureRequestIds: string[]): Observable<string> {
    return this._model.sign(documentId, actualFile, arrayBuffer, certificate, signatureRequestIds);
  }

  verify(file: ArrayBuffer, sign: ArrayBuffer, signatureRequset: ISignatureRequest): Observable<ISignatureVerificationResult> {
    return this._model.verify(file, sign, signatureRequset);
  }

  verifyImportedSignature(file: ArrayBuffer, sign: ArrayBuffer): Observable<IImportedSignatureVerificationResult> {
    return this._model.verifyImportedSignature(file, sign);
  }
  getCertificates(): Observable<ICertificate[]> {
    return this._model.getCertificates();
  }

  canProcessAlgorithms(publicKeyOid: string): boolean {
    return this._model.canProcessAlgorithms(publicKeyOid);
  }

  canProcessSignature(signatureFile: ArrayBuffer): boolean {
    return this._model.canProcessSignature(signatureFile);
  }
}
