/*
Copyright © 2024 ASCON-Design Systems LLC. All rights reserved.
This sample is licensed under the MIT License.
*/

import { Asn1Parser } from "./asn.parser";

const CADESCOM_HASH_ALGORITHM_SHA1 = 0;
const CADESCOM_HASH_ALGORITHM_MD2 = 1;
const CADESCOM_HASH_ALGORITHM_MD4 = 2; 
const CADESCOM_HASH_ALGORITHM_MD5 = 3; 
const CADESCOM_HASH_ALGORITHM_SHA_256 = 4; 
const CADESCOM_HASH_ALGORITHM_SHA_384 = 5;
const CADESCOM_HASH_ALGORITHM_SHA_512 = 6; 
const CADESCOM_HASH_ALGORITHM_CP_GOST_3411 = 100; 
const CADESCOM_HASH_ALGORITHM_CP_GOST_3411_2012_256 = 101; 
const CADESCOM_HASH_ALGORITHM_CP_GOST_3411_2012_512 = 102;
const CADESCOM_HASH_ALGORITHM_CP_GOST_3411_HMAC = 110; 
const CADESCOM_HASH_ALGORITHM_CP_GOST_3411_2012_256_HMAC = 111; 
const CADESCOM_HASH_ALGORITHM_CP_GOST_3411_2012_512_HMAC = 112; 

export class AlgorithmDetector {

  private static _algorithmsOIDs = new Map<string, number>([
    // signatures
    [ '1.3.14.3.2.26', CADESCOM_HASH_ALGORITHM_SHA1 ],
    [ '1.2.840.113549.2.2', CADESCOM_HASH_ALGORITHM_MD2 ],
    [ '1.2.840.113549.2.4', CADESCOM_HASH_ALGORITHM_MD4 ],
    [ '1.2.840.113549.2.5', CADESCOM_HASH_ALGORITHM_MD5 ],
    [ '2.16.840.1.101.3.4.2.1', CADESCOM_HASH_ALGORITHM_SHA_256 ],
    [ '2.16.840.1.101.3.4.2.2', CADESCOM_HASH_ALGORITHM_SHA_384 ],
    [ '2.16.840.1.101.3.4.2.3', CADESCOM_HASH_ALGORITHM_SHA_512 ],
    [ '1.2.643.2.2.9', CADESCOM_HASH_ALGORITHM_CP_GOST_3411 ],
    [ '1.2.643.7.1.1.2.2', CADESCOM_HASH_ALGORITHM_CP_GOST_3411_2012_256 ],
    [ '1.2.643.7.1.1.2.3', CADESCOM_HASH_ALGORITHM_CP_GOST_3411_2012_512 ],
    [ '1.2.643.2.2.10', CADESCOM_HASH_ALGORITHM_CP_GOST_3411_HMAC ],
    [ '1.2.643.7.1.1.4.1', CADESCOM_HASH_ALGORITHM_CP_GOST_3411_2012_256_HMAC ],
    [ '1.2.643.7.1.1.4.2', CADESCOM_HASH_ALGORITHM_CP_GOST_3411_2012_512_HMAC ],
    // public keys
    [ '1.2.643.7.1.1.1.1', CADESCOM_HASH_ALGORITHM_CP_GOST_3411_2012_256],
    [ '1.2.643.7.1.1.1.2', CADESCOM_HASH_ALGORITHM_CP_GOST_3411_2012_512],
    [ '1.2.840.113549.1.1.11', CADESCOM_HASH_ALGORITHM_SHA_256],
    [ '1.2.840.113549.1.1.12', CADESCOM_HASH_ALGORITHM_SHA_384],
    [ '1.2.840.113549.1.1.13', CADESCOM_HASH_ALGORITHM_SHA_512],
    [ '1.2.840.113549.1.1.1', CADESCOM_HASH_ALGORITHM_SHA1],
    [ '1.2.840.113549.1.1.2', CADESCOM_HASH_ALGORITHM_MD2],
    [ '1.2.840.113549.1.1.3', CADESCOM_HASH_ALGORITHM_MD4],
    [ '1.2.840.113549.1.1.4', CADESCOM_HASH_ALGORITHM_MD5],
    [ '1.2.840.113549.1.1.5', CADESCOM_HASH_ALGORITHM_SHA1]
  ]);

  private static get default(): number {
    return CADESCOM_HASH_ALGORITHM_CP_GOST_3411_2012_256;
  }

  static async detectDigestAlgFrom(signOrCertificate: ArrayBuffer | ICryptoProCertificate): Promise<number> {
    // detect from signature
    if ('byteLength' in signOrCertificate && typeof signOrCertificate.byteLength === 'number') {
      const asnObject = Asn1Parser.initAsn(new Uint8Array(signOrCertificate));
      const algorithms = Asn1Parser.searchInTree(asnObject, 'digestAlgorithms');
      if (algorithms) {
        const firstAlgorithm = Asn1Parser.searchInTree(algorithms, 'algorithm');
        return AlgorithmDetector.detect(firstAlgorithm?.content());
      }
    }
    else { // detect from certificate
      const certificate = signOrCertificate as ICryptoProCertificate;
      const value = await this.getCertificatePublicKeyOid(certificate);
      return AlgorithmDetector.detect(value);
    }

    return this.default;
  }

  static async getCertificatePublicKeyOid(certificate: ICryptoProCertificate): Promise<string> {
    const key = await certificate.PublicKey();
    const algorithm = await key.Algorithm;
    const value = await algorithm.Value;
    return value;
  }

  private static detect(text: string | null | undefined): number {
    let algorithm = this.default;
    if (text) {
      this._algorithmsOIDs.forEach((v,k) => {
        if (text.includes(k)) {
          algorithm = v;
          return v;
        }
      });
    }
    return algorithm;
  }
}