/*
Copyright © 2024 ASCON-Design Systems LLC. All rights reserved.
This sample is licensed under the MIT License.
*/

import {
  ICertificate, 
  ISignatureVerificationResult, 
  ISignatureVerificationResultBase, 
  SignatureVerificationStatus, 
  CadesType, 
  ExpectedError, 
  IObjectsRepository, 
  ISignatureRequest, 
  ICryptoProvider, 
  IInitializable, 
  IImportedSignatureVerificationResult 
} from '@pilotdev/pilot-web-sdk';
import { convertToString } from '../utils/utils';
import { CryptoProHashCalculator } from './crypto-pro-hash.calculator';
import { AlgorithmDetector } from './algorithm.detector';
import { CertificateValidator } from './certificate.validator';
import { SignatureTypeConverter } from './signature-type.converter';
import { CryptoProCSPError, ErrorSeverity } from './error-handling/crypto-pro-csp.errors';
import { IAdapter, IErrorConverter } from "./adapter";
import { CryptoProPluginInitializationError, errorMessagePrefixes } from './error-handling/crypto-pro.errors';
import { IParsedSignature, SignatureParser } from './signature.parser';
// @ts-ignore
import { Base64 } from '@lapo/asn1js/base64.js';

import './cadesplugin.js';

export type OriginType = keyof ICryptoProvider | keyof IInitializable;

const CADES_OBJECT_STORE = "CAdESCOM.Store";
const CADES_OBJECT_SIGNER = "CAdESCOM.CPSigner";
const CADES_OBJECT_SIGNED_DATA = "CAdESCOM.CadesSignedData";

declare global {
  interface Window {
    cadesplugin_skip_extension_install: boolean;
  }
}

export class CryptoProAdapter implements IAdapter, IErrorConverter {
  private _eventHandler = this.loadEventHandler.bind(this);
  private _initializationError?: Error;
  private _hashCalculator: CryptoProHashCalculator;
  
  constructor(
    private readonly _certificateValidator: CertificateValidator,
    private readonly loadedCallback: (error?: Error) => void,
    private readonly repository: IObjectsRepository
  ) {
    this._hashCalculator = new CryptoProHashCalculator();
    window.cadesplugin_skip_extension_install = true;
  }
  
  async init(): Promise<void> {
    try {
      this._initializationError = undefined;
      window.addEventListener("message", (event) => {
        this._eventHandler(event)
      }, false);

      await cadesplugin;
      
      window.postMessage("cadesplugin_echo_request", "*");

      this.loadedCallback();
    } catch (err) {
      this._initializationError = new CryptoProPluginInitializationError(`КриптоПро ЭЦП Browser plug-in не установлен или не подключен`);
      throw this.handleError(this._initializationError, "initialize");
    }
  }

  async sign(file: ArrayBuffer, certificate: ICertificate): Promise<string> {

    try {
      this.checkInitialized();
      
      if (!file.byteLength) 
        throw new ExpectedError('Файл пустой, пожалуйста укажите другой файл');
      
      const cryptoCertificate = await this.findCertificate(certificate);
      if (!cryptoCertificate)
        throw new ExpectedError("Не нашлось подходящих сертификатов для подписания файла."); 
      
      const hashData = await this._hashCalculator.calculateHash(file, cryptoCertificate);
      const signature = await this.signHash(hashData, cryptoCertificate);
      return signature;

    } catch (error) {
      const cryptoProError = this.handleCryptoProError(error);
      throw this.handleError(cryptoProError, "sign");
    }
  }


  async verify(file: ArrayBuffer, sign: ArrayBuffer, signatureRequset: ISignatureRequest): Promise<ISignatureVerificationResult> {
    const verificationResult = { verificationStatus: SignatureVerificationStatus.Unknown } as ISignatureVerificationResult;
    try {
      this.checkInitialized();
      
      const signBase64 = convertToString(sign);
      const parsedSignature = SignatureParser.parse(signBase64);
      if (parsedSignature.subject && parsedSignature.subject.commonName) {
        const orgUnit = this.repository.getOrganisationUnit(signatureRequset.positionId);
        verificationResult.signerName = orgUnit.title == undefined ? parsedSignature.subject.commonName : `${parsedSignature.subject.commonName} (${orgUnit.title})`;
      }
      
      verificationResult.signDate = this.getSigningDate(parsedSignature, signatureRequset.lastSignCadesType);

      const unarmoredSignature = Base64.unarmor(signBase64)
      const hashedData = await this._hashCalculator.calculateHash(file, unarmoredSignature);
      verificationResult.verificationStatus = await this.verifyHash(hashedData, signBase64, signatureRequset.lastSignCadesType);

    } catch (err) {
      const cryptoProError = this.handleCryptoProError(err);
      this.handleVerificationError(cryptoProError, verificationResult);
    } finally {
      return verificationResult;
    }
  }

  async verifyImportedSignature(file: ArrayBuffer, sign: ArrayBuffer): Promise<IImportedSignatureVerificationResult> {
    const verificationResult = { verificationStatus: SignatureVerificationStatus.Unknown } as IImportedSignatureVerificationResult;
    try {
      this.checkInitialized();

      const signBase64 = convertToString(sign);
      const parsedSignature = SignatureParser.parse(signBase64);
      if (parsedSignature.subject && parsedSignature.subject.commonName) {
        verificationResult.signerName = parsedSignature.subject.commonName;
      }
      verificationResult.publicKeyOid = parsedSignature.signatureAlgorithm;
      verificationResult.cadesType = parsedSignature.cadesType;
      const hashedData = await this._hashCalculator.calculateHash(file, sign);
      verificationResult.verificationStatus = await this.verifyHash(hashedData, signBase64, parsedSignature.cadesType);

    } catch (err) {
      const cryptoProError = this.handleCryptoProError(err);
      this.handleVerificationError(cryptoProError, verificationResult); 
    } finally {
      return verificationResult;
    }
  }

  async getCertificates(): Promise<ICertificate[]> {
    try {
      this.checkInitialized();
      
      const store = await this.openStore();
      const certificates = await store.Certificates;
      const count = await certificates?.Count;
      const certificatesInfo: ICertificate[] = [];
      
      for(let i = 1; i <= count; i++) {
        const cert = await certificates.Item(i);
        
        const isValid = await this._certificateValidator.isValidCertificate(cert);
        if (!isValid)
          continue;
        
        const certificate = await this.convertCertificate(cert);
        certificatesInfo.push(certificate);
      }
      
      await store.Close();
      return certificatesInfo;

    } catch(error) {
      const cryptoProError = this.handleCryptoProError(error);
      throw this.handleError(cryptoProError, "getCertificates");
    }
  }

  convertToError(err: unknown): CryptoProCSPError | null {
    try {
      this.checkInitialized();
    } catch (error) {
      return null;
    }

    if (!this.isConvertableError(err))
      return null;

    const errorResult = err as ICadesException;
    const message = cadesplugin.getLastError(errorResult);

    return new CryptoProCSPError(message);
  }

  private isConvertableError(err: unknown): boolean {
    return (!!err && typeof err === "object" && "message" in err && "requestid" in err && "type" in err);
  }

  private async signHash(oHashedData: ICryptoProHashedData, certificate: ICryptoProCertificate): Promise<string> { 
    const signer = await cadesplugin.CreateObjectAsync(CADES_OBJECT_SIGNER) as ICryptoProSigner;
    await signer.propset_Certificate(certificate);
    await signer.propset_CheckCertificate(true);
    // http://pki.skbkontur.ru/tsp2012/tsp.srf
    // http://pki.tax.gov.ru/tsp/tsp.srf
    // http://testca.cryptopro.ru/tsp/

    const signedData = await cadesplugin.CreateObjectAsync(CADES_OBJECT_SIGNED_DATA) as ICryptoProSignedData;
    await signedData.propset_ContentEncoding(cadesplugin.CADESCOM_BASE64_TO_BINARY);

    const signature = await signedData.SignHash(oHashedData, signer, cadesplugin.CADESCOM_CADES_BES);
    return signature;
  }


  private async verifyHash(hashedData: ICryptoProHashedData, sign: string, cadesType: CadesType): Promise<SignatureVerificationStatus> {
    
    const signedData = await cadesplugin.CreateObjectAsync(CADES_OBJECT_SIGNED_DATA) as ICryptoProSignedData;
    const cryptoProSignatureType = SignatureTypeConverter.convert(cadesType);
    await signedData.VerifyHash(hashedData, sign, cryptoProSignatureType);
    const signers = await signedData.Signers;
    const signer = await signers.Item(1);
    const certificate = await signer.Certificate;
    const valid = await certificate.IsValid();
    const isCertificateValid = await valid.Result;
    const signStatus = await signer.SignatureStatus;
    const isValidSignStatus = await signStatus.IsValid;
    
    if (isValidSignStatus && isCertificateValid)
      return SignatureVerificationStatus.Valid;

    if (isValidSignStatus && !isCertificateValid)
      return SignatureVerificationStatus.ValidWithWarnings;

    return SignatureVerificationStatus.Invalid;
  }

  private getSigningDate(signature: IParsedSignature, cadesType: CadesType): string | undefined {
    if (cadesType >= CadesType.CadesT) {
      if (signature.timeStampDate)
        return (new Date(signature.timeStampDate)).toISOString();
      else 
        return undefined;
    }
    
    if (signature.signingDate)
      return (new Date(signature.signingDate)).toISOString();

    return undefined;
  }

  private async findCertificate(certificate: ICertificate): Promise<ICryptoProCertificate | null> {
    const store = await this.openStore();
    const certificates = await store.Certificates;
    if (!certificates) {
      await store.Close();
      return null;
    }

    const count = await certificates.Count;
    if (count === 0) {
      await store.Close();
      return null;
    }

    for (let index = 1; index <= count; index++) {
      const cryptoCertificate = await certificates.Item(index);
      const thumbprint = await cryptoCertificate.Thumbprint;
      if (certificate.thumbprint === thumbprint) {
        await store.Close();
        return cryptoCertificate;
      }
    }

    return null;
  }

  private loadEventHandler(event: MessageEvent): void {
    if (event.data == "cadesplugin_loaded") {
      // прикладной код
      this._initializationError = undefined;
      window.removeEventListener('message', this._eventHandler);
      this.loadedCallback();
    } else if(event.data == "cadesplugin_load_error") {
      // сообщение об ошибке
      this._initializationError = new CryptoProPluginInitializationError(`Общая ошибка инициализации плагина ${event.data}`);
      throw this._initializationError;
    }
  }

  private async openStore(name: string = cadesplugin.CAPICOM_MY_STORE): Promise<ICryptoProStore> {
    const store = <ICryptoProStore> await cadesplugin.CreateObjectAsync(CADES_OBJECT_STORE);
    await store.Open(cadesplugin.CAPICOM_CURRENT_USER_STORE, name, cadesplugin.CAPICOM_STORE_OPEN_MAXIMUM_ALLOWED);
    return store;
  }

  private checkInitialized(): void {
    if (this._initializationError) {
      throw this._initializationError;
    }
  }

  private async convertCertificate(certificate: ICryptoProCertificate): Promise<ICertificate> {
    const thumbprint = await certificate.Thumbprint;
    const subject = await certificate.SubjectName;
    const issuer = await certificate.IssuerName;
    const validFromDate = await certificate.ValidFromDate;
    const validToDate = await certificate.ValidToDate;
    const publicKeyOid = await AlgorithmDetector.getCertificatePublicKeyOid(certificate);

    return {
      thumbprint: thumbprint,
      subject,
      issuer,
      validFromDate,
      validToDate,
      publicKeyOid: publicKeyOid
    };
  }

  private handleVerificationError(error: Error, verificationResult: ISignatureVerificationResultBase) {
    verificationResult.verificationStatus = SignatureVerificationStatus.Error;
    
    if (error instanceof CryptoProPluginInitializationError) {
      verificationResult.verificationStatus = SignatureVerificationStatus.CannotBeChecked
    }
    
    if (error instanceof CryptoProCSPError && error.severity === ErrorSeverity.Warning) {
      verificationResult.verificationStatus = SignatureVerificationStatus.ValidWithWarnings;
    }


    verificationResult.error =
    error && typeof error === "object" && "message" in error
        ? (error as { message?: string }).message
        : errorMessagePrefixes["unexpected"];
  } 
  
  private handleCryptoProError(error: unknown): Error {
    let convertedError = this.convertToError(error);

    if (convertedError) {
      return convertedError;
    }

    if (error instanceof CryptoProPluginInitializationError) {
      return error;
    }

    if (error instanceof Error) {
      return error;
    }

    if (typeof error === "string") {
      return new Error(error);
    }

    return new Error(JSON.stringify(error));
  }

  private handleError(error: unknown, origin: OriginType): Error {
    if (!error || !(typeof error === "object" && "message" in error))
      return new Error(`${errorMessagePrefixes["unexpected"]}`);

    const originPrefix = errorMessagePrefixes[origin];
    const message = originPrefix.length > 0 ? `${originPrefix} ${error.message}` : `${error.message}`;

    if (error instanceof ExpectedError) {
      return error;
    }

    if (error instanceof CryptoProPluginInitializationError) {
      return new ExpectedError(message);
    }

    if (error instanceof CryptoProCSPError) {
      switch (error.severity) {
        case ErrorSeverity.NonError:
          return new ExpectedError(message, true);
        case ErrorSeverity.Warning:
          return new ExpectedError(message);
        case ErrorSeverity.Error:
          return new Error(message);
      }
    }

    return new Error(message);
  }
}
