/**
 * Copyright (c) 2024 ASCON-Design Systems LLC. All rights reserved.
 * This sample is licensed under the MIT License.
 */
import {
  ApplicationRef,
  ComponentRef,
  createComponent, importProvidersFrom, Inject,
  Injectable, InjectionToken,
  NgZone,
} from '@angular/core';
import { createApplication } from '@angular/platform-browser';

import {
  DialogContext,
  IDisposable, IObjectsRepository,
  IOpenspaceView
} from '@pilotdev/pilot-web-sdk';

import { ObjectInfoComponent } from "../../components/object-info/object-info.component";
import { ObjectInfoViewId } from "./consts";
import { provideAnimations } from "@angular/platform-browser/animations";
import { HttpClient, HttpClientModule } from "@angular/common/http";
import { TranslateLoader, TranslateModule } from "@ngx-translate/core";
import { createTranslateLoader } from "../../helpers/translation-loader";
import { ObjectInfoContainerComponent } from "../../components/object-info-container/object-info-container.component";

export const ASSETS_URL = new InjectionToken<string>('assets_url');

@Injectable({ providedIn: 'root' })
export class ObjectInfoPage implements IOpenspaceView<DialogContext>, IDisposable
{
  private _rootElement: HTMLElement | undefined;
  private _appRef: ApplicationRef | undefined;
  private _componentRef: ComponentRef<ObjectInfoContainerComponent> | undefined;

  constructor(@Inject('IObjectsRepository') private _objectsRepository: IObjectsRepository) {
  }

  getViewId(): string {
    return ObjectInfoViewId;
  }

  getView(context: DialogContext): HTMLElement | undefined {
    if (!this._rootElement) {
      this._rootElement = document.createElement('div');
    }

    if (!this._appRef) {
      createApplication({
        providers: [
          { provide: "ObjectsViewContext", useValue: context.data },
          { provide: ASSETS_URL, useValue: __webpack_public_path__ },
          { provide: "IObjectsRepository", useValue: this._objectsRepository },
          provideAnimations(),
          importProvidersFrom([HttpClientModule, TranslateModule.forRoot({
            loader: {
              provide: TranslateLoader, useFactory: createTranslateLoader,
              deps: [ HttpClient, ASSETS_URL]
            }
          })])
        ],
      }).then((appRef: ApplicationRef) => {
        this._appRef = appRef;

        const zone = appRef.injector.get(NgZone);
        zone.run(() => {
          this._componentRef = createComponent(ObjectInfoContainerComponent, {
            environmentInjector: appRef.injector,
            hostElement: this._rootElement,
          });

          appRef.attachView(this._componentRef.hostView);
        });
      });
    }

    return this._rootElement;
  }

  dispose(): void {
    if (this._componentRef) {
      this._componentRef.destroy();
      this._componentRef = undefined;
    }

    if (this._appRef) {
      this._appRef.destroy();
      this._appRef = undefined;
    }

    if (this._rootElement) {
      this._rootElement = undefined;
    }
  }
}
