﻿/*
  Copyright © 2025 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/

using Org.BouncyCastle.Asn1.X509;
using Org.BouncyCastle.Cms;
using Org.BouncyCastle.X509;
using Pilot.WebServer.SDK;

namespace PdfStamper.Certificates;

internal class CertificateReader
{
    private readonly IFileLoader _fileLoader;

    public CertificateReader(IFileLoader fileLoader)
    {
        _fileLoader = fileLoader;
    }
    public IEnumerable<CertificateInfo> GetCertificateInfoList(PdfDocumentContext context)
    {
        var snapshot = Utils.GetFilesSnapshot(context);
        if (snapshot == null)
            return Array.Empty<CertificateInfo>();

        var signatureRequests = snapshot.Files.SelectMany(x => x.SignatureRequests).Where(y => y.Signs.Any()).ToList();

        if (!signatureRequests.Any())
            return Array.Empty<CertificateInfo>();

        var certificateInfoList = new List<CertificateInfo>();
        foreach (var request in signatureRequests)
        {
            var signFileId = request.Signs.LastOrDefault();
            if (string.IsNullOrEmpty(signFileId))
                continue;

            var signatureFile = snapshot.Files.FirstOrDefault(x => x.Id.Equals(Guid.Parse(signFileId)));
            if (signatureFile == null)
                continue;

            var signature = _fileLoader.Download(signatureFile.Id, signatureFile.Size);
            var certificateInfos = LoadCertificates(signature, signFileId, request.Role);
            certificateInfoList.AddRange(certificateInfos);
        }
        return certificateInfoList;
    }

    private string GetCN(X509Name name)
    {
        return name.GetValueList(X509Name.CN).FirstOrDefault()?.ToString() ?? string.Empty;
    }

    private string GetThumbprint(X509Certificate certificate)
    {
        // В .NET свойство X509Certificate2.Thumbprint исторически рассчитывается как SHA-1-хеш от
        // DER-кодировки сертификата — просто потому, что так сложилось в Windows/API: “thumbprint”
        // изначально задумывался как короткий (20-байтовый) отпечаток, удобный для идентификации
        // сертификата. По-этому ничего не будем придумывать и используем SHA1.
        using var sha1 = System.Security.Cryptography.SHA1.Create();
        byte[] hash = sha1.ComputeHash(certificate.GetEncoded());
        // BitConverter даёт вида "AB-CD-01...", уберём дефисы
        return BitConverter.ToString(hash).Replace("-", "").ToUpperInvariant();
    }

    private IEnumerable<CertificateInfo> LoadCertificates(byte[] certificateData, string signId, string role)
    {
        // Здесь используем библиотеку BouncyCastle.Cryptography потому, что в 
        // .Net нет обработки алгоритмов для ГОСТ.
        
        // Изначально все подписи из пилота приходят в Base64, по-этому, 
        // чтобы получить реальный массив байт самого сертификата
        // надо сконвертировать его из base64.
        var rawCertificateFile = ConvertFromBase64(certificateData);
        var cms = new CmsSignedData(rawCertificateFile);
        
        // Получаем информацию о подписантах
        var signers = cms.GetSignerInfos();
        foreach (SignerInformation signer in signers.GetSigners())
        {
            var signerId = signer.SignerID;
            // Получаем сертификат подписанта
            var certStore = cms.GetCertificates();
            var certs = certStore.EnumerateMatches(signerId);

            foreach (var cert in certs)
            {
                var certificateInfo = new CertificateInfo
                {
                    Id = signId,
                    SignerRole = role,
                    Subject = GetCN(cert.SubjectDN),
                    Thumbprint = GetThumbprint(cert),
                    ValidFrom = cert.NotBefore.ToShortDateString(),
                    ValidTo = cert.NotAfter.ToShortDateString()
                };

                var tst = Asn1Helper.GetTimestampToken(signer);
                if (tst != null)
                {
                    var tstCertificate = Asn1Helper.GetTimeStampCertificate(tst);
                    if (tstCertificate != null)
                    {
                        certificateInfo.ValidFrom = tstCertificate.NotBefore.ToShortDateString();
                        certificateInfo.ValidTo = tstCertificate.NotAfter.ToShortDateString();
                    }
                }

                yield return certificateInfo;
            }
        }
    }

    private byte[] ConvertFromBase64(byte[] bytes)
    {
        var str = System.Text.Encoding.Default.GetString(bytes);
        return Convert.FromBase64String(str);
    }
}
