﻿/*
  Copyright © 2025 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using PdfSharp.Drawing;
using PdfStamper.Certificates;
using PdfStamper.Settings;
using Pilot.WebServer.SDK;
using System.ComponentModel.Composition;

namespace PdfStamper;

[Export(typeof(IPdfStamper))]
public class PdfStamper : IPdfStamper
{
    private readonly CertificateReader _certificateReader;
    private readonly SettingsManager _settingsManager;
    private readonly FileStampProvider _fileStampProvider;

    private readonly XSize _stampSize = new XSize(XUnit.FromMillimeter(77).Point, XUnit.FromMillimeter(25).Point); // Может тоже настройки?

    [ImportingConstructor]
    public PdfStamper(IFileLoader fileLoader, ISettingsStore settingsStore, IModifierProvider modifierProvider)
    {
        _certificateReader = new CertificateReader(fileLoader);
        _settingsManager = new SettingsManager(settingsStore);
        _fileStampProvider = new FileStampProvider(fileLoader, modifierProvider);
    }

    public void AddStamps(Stream stream, PdfStamperMode mode, PdfDocumentContext context)
    {
        if (context.Document == null)
            return;

        var certificateInfoList = _certificateReader.GetCertificateInfoList(context);
        if (!certificateInfoList.Any())
            return;

        var settingsModel = _settingsManager.GetStampSettings(context.Document);
        using var stamper = new StampCreator(stream, mode, settingsModel);

        var pagesCount = stamper.GetPagesCount();
        if (!settingsModel.IsOnLastPage && pagesCount < settingsModel.PageNumber)
            return;

        var pageNumber = settingsModel.IsOnLastPage ? pagesCount - 1 : settingsModel.PageNumber - 1;
        var defaultStartPosition = settingsModel.CalculateFirstPosition(stamper.GetPageSize(pageNumber));
        var offsetPrevX = XUnit.FromMillimeter(settingsModel.OffsetPreviousX);
        var offsetPrevY = XUnit.FromMillimeter(settingsModel.OffsetPreviousY);
        
        foreach (var certificateInfo in certificateInfoList)
        {
            var stampPosition = _fileStampProvider.GetStampPosition(context, certificateInfo.Id!);
            var stampType = GetStampType(certificateInfo, settingsModel);
            stamper.AddStamp(pageNumber, certificateInfo, stampPosition ?? defaultStartPosition, _stampSize, stampType);
            
            if (stampPosition == null)
            {
                defaultStartPosition.X += offsetPrevX.Point;
                defaultStartPosition.Y += offsetPrevY.Point;
            }
        }

        stamper.SaveStamps();
    }

    public bool CanAddStamps(PdfDocumentContext context)
    {
        if (context?.Document == null)
            return false;

        var snapshot = Utils.GetFilesSnapshot(context);
        if (snapshot == null)
            return false;

        var signatureRequests = snapshot.Files
            .SelectMany(x => x.SignatureRequests)
            .Where(y => y.Signs.Any())
            .Where(x => x.LastSignCadesType != Pilot.WebServer.SDK.Data.CadesType.Unknown && 
                        x.LastSignCadesType != Pilot.WebServer.SDK.Data.CadesType.NotCades);
        return signatureRequests.Any();
    }

    public void OnStampPositionChanged(StampPositionArgs args)
    {
        _fileStampProvider.UpdateStampPosition(args);
    }

    private StampType GetStampType(CertificateInfo certificateInfo, SettingsModel settingsModel)
    {
        var inProduction = certificateInfo.SignerRole != null &&
                           certificateInfo.SignerRole.Equals(settingsModel.IntoProductionRole, StringComparison.CurrentCultureIgnoreCase);
        return inProduction ? StampType.IntoProduction : StampType.Signature;
    }
}
