/**
 * Copyright (c) 2024 ASCON-Design Systems LLC. All rights reserved.
 * This sample is licensed under the MIT License.
 */
import {
  ApplicationRef,
  ComponentRef,
  createComponent,
  Inject,
  Injectable,
  InjectionToken,
  NgZone,
} from '@angular/core';
import { createApplication } from '@angular/platform-browser';
import { provideNoopAnimations } from '@angular/platform-browser/animations';
import {
  IDisposable,
  IOpenspaceView,
  IThemeService,
  ObjectsViewContext,
} from '@pilotdev/pilot-web-sdk';

import { TableDemoComponent } from '../../table-demo/table-demo.component';
import { TableTabId } from '../document.tabs';

@Injectable({ providedIn: 'root' })
export class TableDemoSample
  implements IOpenspaceView<ObjectsViewContext>, IDisposable
{
  private _rootElement: HTMLElement | undefined;
  private _appRef: ApplicationRef | undefined;
  private _componentRef: ComponentRef<TableDemoComponent> | undefined;

  constructor(@Inject('IThemeService') private _themeService: IThemeService) {}

  getViewId(): string {
    return TableTabId;
  }

  getView(context: ObjectsViewContext): HTMLElement | undefined {
    if (!this._rootElement) {
      this._rootElement = document.createElement('div');
      this._rootElement.id = context.selectedObjects[0].id;
    }

    if (!this._appRef) {
      createApplication({
        providers: [
          { provide: 'IThemeService', useValue: this._themeService },
          provideNoopAnimations(),
        ],
      }).then((appRef: ApplicationRef) => {
        this._appRef = appRef;

        const zone = appRef.injector.get(NgZone);
        zone.run(() => {
          this._componentRef = createComponent(TableDemoComponent, {
            environmentInjector: appRef.injector,
            hostElement: this._rootElement,
          });

          appRef.attachView(this._componentRef.hostView);
        });
      });
    }

    return this._rootElement;
  }

  dispose(): void {
    if (this._componentRef) {
      this._componentRef.destroy();
      this._componentRef = undefined;
    }

    if (this._appRef) {
      this._appRef.destroy();
      this._appRef = undefined;
    }

    if (this._rootElement) {
      this._rootElement = undefined;
    }
  }
}
