/// <reference types="@pilotdev/pilot-web-2d" />
import { IMenu, IMenuBuilder, DocumentAnnotationsListContext,IRenderContextProvider, IObjectsRepository, AccessLevel,AttributePermissionInfo, IAnnotationContainerObject, IAnnotationContainer, IModifierProvider, IInitializable, InjectionSource } from "@pilotdev/pilot-web-sdk";
import { ClickEvent, ClickNotifier } from "./clickNotifier";
import { Unsubscribable } from "rxjs";


export class AnnotationPageChanger extends IMenu<DocumentAnnotationsListContext> implements IInitializable {

  static readonly RED_PENCIL = 'RedPencil';
  static readonly TEXT_NOTE = 'TextNote';
  static readonly STICKY_NOTE = 'TextStickyNote';
  static readonly ANNOTATION_ATTRIBUTE_NAME = "annotation";
  
  private _subscription?: Unsubscribable;
  private _annotationToCut?: IAnnotationContainerObject;

  private _objectsRepository!: IObjectsRepository;
  private _renderContextProvider!: IRenderContextProvider;
  private _modifierProvider!: IModifierProvider;

  constructor() {

    super();
    if (window.clickNotifier == null)
      window.clickNotifier = new ClickNotifier();
  }

  initialize(injectionSource: InjectionSource): void {
      this._objectsRepository = injectionSource.objectsRepository;
      this._renderContextProvider = injectionSource.renderContextProvider;
      this._modifierProvider = injectionSource.modifierProvider;
  }

  build(builder: IMenuBuilder, context: DocumentAnnotationsListContext): void {

    if (!this.canCutAnnotation(context))
      return;

    let count = builder.count;
    builder.addItem("apc menu", count).withHeader("Перенести замечание на другую страницу")
  }

  onMenuItemClick(name: string, context: DocumentAnnotationsListContext): void {

    this._annotationToCut = context.selectedAnnotations[0];
    this._subscription = window.clickNotifier.onClicked.subscribe(this.onDocumentClick.bind(this));
  }


  onDocumentClick(event: ClickEvent): void {

    if (!this._annotationToCut)
      return;

    const newAnnotation = {
      annotationId: this._annotationToCut.container.annotationId,
      version: this._annotationToCut.container.version,
      positionX:  event.point.x,
      positionY: event.point.y,
      pageNumber: event.page,
      data:  this._annotationToCut.container.annotationId,
      kind:  this._annotationToCut.container.annotationId
    } as IAnnotationContainer;

    const modifier = this._modifierProvider.newModifier();
    const builder = modifier.edit(newAnnotation.annotationId!);
    builder.addAnnotationContainer(newAnnotation, this._annotationToCut.attributes);
    modifier.apply().subscribe();

    this._subscription?.unsubscribe();
    this._annotationToCut = undefined;
  }

  private canCutAnnotation(context: DocumentAnnotationsListContext): boolean {

    if (context.selectedAnnotations.length != 1) //can use only on one annotation
      return false;

    const annotation = context.selectedAnnotations[0]; 
    const currentPerson = this._objectsRepository.getCurrentPerson();
    const access = annotation.source!.accessLevel;
    
    if ((access & AccessLevel.Edit) !== AccessLevel.Edit) //should have edit access for object
      return false;

    const attr = annotation.source.type.attributes.find(x => x.name == AnnotationPageChanger.ANNOTATION_ATTRIBUTE_NAME);
    const pi = new AttributePermissionInfo(attr!.editableForPositionsSource);
    if (pi.extractAllOrgUnits(annotation.source).filter(u => currentPerson.allOrgUnits.includes(u)).length === 0)
      return false; //should have edit access for attribute

    const renderContext = this._renderContextProvider.getRenderContext();
    if (renderContext.selectedVersion !== renderContext.dataObject.actualFileSnapshot.created) //document should be open in actual version
      return false;

    
    if (annotation.container.version !== renderContext.selectedVersion) //annotation should be in actual version
      return false;

    
    if (annotation.container.kind == AnnotationPageChanger.STICKY_NOTE) //can't move text sticky annotation
      return false;
    

    return true;
  }
  
}



