/**
 * Copyright (c) 2024 ASCON-Design Systems LLC. All rights reserved.
 * This sample is licensed under the MIT License.
 */
import { ApplicationRef, ComponentRef, createComponent, Inject, Injectable, NgZone } from '@angular/core';
import { createApplication } from '@angular/platform-browser';

import { BimElementPanelContext, IDisposable, IOpenspaceView, IThemeService } from '@pilotdev/pilot-web-sdk';

import { ElementPanelTabId } from '../ids';
import { ElementViewComponent } from '../../components/element-view/element-view.component';
import { THEME_SERVICE } from '../../tokens';

@Injectable({ providedIn: 'root' })
export class BimElementTabView
  implements IOpenspaceView<BimElementPanelContext>, IDisposable
{
  private _rootElement: HTMLElement | undefined;
  private _appRef: ApplicationRef | undefined;
  private _componentRef: ComponentRef<ElementViewComponent> | undefined;


  constructor(
    @Inject('IThemeService')
    private readonly _themeService: IThemeService
  ) {
  }

  getViewId(): string {
    return ElementPanelTabId;
  }

  getView(context: BimElementPanelContext): HTMLElement | undefined {
    if (!this._rootElement) {
      this._rootElement = document.createElement('div');
    }

    if (!this._appRef) {
      createApplication({
        providers: [
          { provide: THEME_SERVICE, useValue: this._themeService },
          { provide: BimElementPanelContext, useValue: context }
        ],
      }).then((appRef: ApplicationRef) => {
        this._appRef = appRef;

        const zone = appRef.injector.get(NgZone);
        zone.run(() => {
          this._componentRef = createComponent(ElementViewComponent, {
            environmentInjector: appRef.injector,
            hostElement: this._rootElement,
          });

          appRef.attachView(this._componentRef.hostView);
        });
      });
    }

    return this._rootElement;
  }

  dispose(): void {
    if (this._componentRef) {
      this._componentRef.destroy();
      this._componentRef = undefined;
    }

    if (this._appRef) {
      this._appRef.destroy();
      this._appRef = undefined;
    }

    if (this._rootElement) {
      this._rootElement = undefined;
    }
  }
}
