/**
 * Copyright (c) 2024 ASCON-Design Systems LLC. All rights reserved.
 * This sample is licensed under the MIT License.
 */
import { ApplicationRef, ComponentRef, createComponent, Inject, Injectable, NgZone } from '@angular/core';
import { createApplication } from '@angular/platform-browser';

import { BimRightPanelContext, IDisposable, IOpenspaceView, IBimFeatures, IThemeService } from '@pilotdev/pilot-web-sdk';

import { RightPanelTabId } from '../ids';
import { RightPanelViewComponent } from '../../components/right-panel-view/right-panel-view.component';
import { BIM_FEATURES, THEME_SERVICE } from '../../tokens';

@Injectable({ providedIn: 'root' })
export class BimRightTabView
  implements IOpenspaceView<BimRightPanelContext>, IDisposable
{
  private _rootElement: HTMLElement | undefined;
  private _appRef: ApplicationRef | undefined;
  private _componentRef: ComponentRef<RightPanelViewComponent> | undefined;

  constructor(
    @Inject('IThemeService')
    private readonly _themeService: IThemeService,
    @Inject("IBimFeatures")
    private readonly _bimFeatures: IBimFeatures
  ) {
  }

  getViewId(): string {
    return RightPanelTabId;
  }

  getView(context: BimRightPanelContext): HTMLElement | undefined {
    if (!this._rootElement) {
      this._rootElement = document.createElement('div');
    }

    if (!this._appRef) {
      createApplication({
        providers: [
          { provide: BimRightPanelContext, useValue: context },
          { provide: THEME_SERVICE, useValue: this._themeService },
          { provide: BIM_FEATURES, useValue: this._bimFeatures }
        ],
      }).then((appRef: ApplicationRef) => {
        this._appRef = appRef;

        const zone = appRef.injector.get(NgZone);
        zone.run(() => {
          this._componentRef = createComponent(RightPanelViewComponent, {
            environmentInjector: appRef.injector,
            hostElement: this._rootElement,
          });

          appRef.attachView(this._componentRef.hostView);
        });
      });
    }

    return this._rootElement;
  }

  dispose(): void {
    if (this._componentRef) {
      this._componentRef.destroy();
      this._componentRef = undefined;
    }

    if (this._appRef) {
      this._appRef.destroy();
      this._appRef = undefined;
    }

    if (this._rootElement) {
      this._rootElement = undefined;
    }
  }
}
