//@ts-ignore
import * as rs from "jsrsasign";

import { ICertificate, IImportedSignatureVerificationResult, ISignatureVerificationResult, SignatureVerificationStatus } from "@pilotdev/pilot-web-sdk";
import { CertificatesStore } from "./certificates.store";
import { convertToString, getCNPart, toJsDate, toJsISODate } from "./utils";
import { paramOrg } from './params';
import { CadesType } from "@pilotdev/pilot-web-sdk/";
import { IAdapter } from "./adapter.inteface";
import { X509 } from "jsrsasign";

export class JsRsaSignAdapter implements IAdapter {

  static readonly signatureAlgorithm = "SHA1withRSA";

  constructor(private readonly _certificatesStore: CertificatesStore) {
  }

  getCertificates(): ICertificate[] {
    try {
      return this._certificatesStore.getCertificates();
    } catch (error) {
      console.error(error);
      return [];
    }
  }

  canProcessSignature(signatureFile: ArrayBuffer): boolean {
    try {
      const signatureBase64 = convertToString(signatureFile);
      const signatureHex = rs.b64tohex(signatureBase64);
      const parser = new (<any>rs.KJUR.asn1.cms).CMSParser();
      const cmsData = parser.getCMSSignedData(signatureHex);
      const sigAlg = cmsData.sinfos[0].sigalg;
      return sigAlg == JsRsaSignAdapter.signatureAlgorithm;
    } catch (error) {
      return false;
    }
  }


  sign(file: ArrayBuffer, certificate: ICertificate): string {
    try {
      return this.signInternal(file, certificate); 
    } catch (error) {
      console.error(error);
      return "";
    }
  }

  verify(file: ArrayBuffer, sign: ArrayBuffer, signatureType: CadesType): ISignatureVerificationResult {
    const result = {} as ISignatureVerificationResult;
    try {
         // initialize for signature validation
        const signatureBase64 = convertToString(sign);
        const signatureHex = rs.b64tohex(signatureBase64);
        const parser = new (<any>rs.KJUR.asn1.cms).CMSParser();
        const cmsData = parser.getCMSSignedData(signatureHex);
        const certPEM = cmsData.certs.array[0];

        // знаем только о нашей структуре подписи
        const signHex = cmsData.sinfos[0].sighex;

        // verify
        const sig = new rs.KJUR.crypto.Signature({ "alg": JsRsaSignAdapter.signatureAlgorithm });
        // initialize for signature validation
        sig.init(certPEM);
        // signer's certificate
        // update data
        const fileBase64 = convertToString(file);
        sig.updateString(fileBase64);
        // get certificate ingo
        const certReader = new X509();
        certReader.readCertPEM(certPEM);

        const signingTime = cmsData.sinfos[0].sattrs.array[1].str;
        result.signDate = toJsISODate(signingTime);
        result.signerName = getCNPart(certReader.getSubjectString());
        result.issuerName = getCNPart(certReader.getIssuerString());

        // verify signature
        const isValid = sig.verify(signHex);
        result.verificationStatus = isValid ? SignatureVerificationStatus.Valid : SignatureVerificationStatus.Invalid;
    } catch (error) {
      console.error(error);
      result.error = (error as Error)?.message;
      result.verificationStatus = SignatureVerificationStatus.Error;
    } finally {
      return result;
    }
  }

  verifyImportedSignature(file: ArrayBuffer, sign: ArrayBuffer): IImportedSignatureVerificationResult {
    const result = {} as IImportedSignatureVerificationResult;
    try {
         // initialize for signature validation
        const signatureBase64 = convertToString(sign);
        const signatureHex = rs.b64tohex(signatureBase64);
        const parser = new (<any>rs.KJUR.asn1.cms).CMSParser();
        const cmsData = parser.getCMSSignedData(signatureHex);
        const certPEM = cmsData.certs.array[0];

        // знаем только о нашей структуре подписи
        const signHex = cmsData.sinfos[0].sighex;

        // verify
        const sig = new rs.KJUR.crypto.Signature({ "alg": "SHA1withRSA" });
        // initialize for signature validation
        sig.init(certPEM);
        // signer's certificate
        // update data
        const fileBase64 = convertToString(file);
        sig.updateString(fileBase64);
        // get certificate ingo
        const certReader = new X509();
        certReader.readCertPEM(certPEM);

        result.signerName = getCNPart(certReader.getSubjectString());
        result.publicKeyOid = JsRsaSignAdapter.signatureAlgorithm;
        result.cadesType = CadesType.CadesBes;
        // verify signature
        const isValid = sig.verify(signHex);
        result.verificationStatus = isValid ? SignatureVerificationStatus.Valid : SignatureVerificationStatus.Invalid;

    } catch (error) {
      console.error(error);
      result.error = (error as Error)?.message;
      result.verificationStatus = SignatureVerificationStatus.Error;
    } finally {
      return result;
    }
  }

  private signInternal(file: ArrayBuffer, certificate: ICertificate): string {
    // initialize
    const fileBase64 = convertToString(file);
    const param = JSON.parse(JSON.stringify(paramOrg)); // deepCopy
    const certPEM = this._certificatesStore.getCertificate(certificate.subject);
    const prvKeyPEM = this._certificatesStore.getPrivateKey(certificate.subject);

    // Sign
    const sig = new rs.KJUR.crypto.Signature({"alg": JsRsaSignAdapter.signatureAlgorithm});
    // initialize for signature generation
    sig.init(prvKeyPEM ?? "");   // rsaPrivateKey of RSAKey object
    // update data
    sig.updateString(fileBase64);
    // calculate signature
    const sigValueHex = sig.sign();

    // Create SignedData
    param.certs = [certPEM];
    param.econtent.content = { str: sigValueHex };

    const sinfo = param.sinfos[0];
    sinfo.id = {type: 'isssn', cert: certPEM};
    sinfo.signkey = prvKeyPEM;
    sinfo.sighex = sigValueHex;

    const sattrs = sinfo.sattrs.array;
    sattrs.push({attr: "signingTime"});
    sattrs.push({attr: "signingCertificate", array:[certPEM]});
    sattrs.push({attr: "signingCertificateV2", array:[certPEM]});

    const sd = new rs.KJUR.asn1.cms.SignedData();
    sd.params = param;
    const hCmsSignedData = sd.getContentInfoEncodedHex();
    return rs.hextob64(hCmsSignedData);
  }
}