/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/

import { Observable, of, from, firstValueFrom,tap } from 'rxjs';
import { ICertificate, ICryptoProvider, IFile, IImportedSignatureVerificationResult, IInitializable, InjectionSource, ISignatureRequest, ISignatureVerificationResult } from '@pilotdev/pilot-web-sdk'
import { JsRsaSignAdapter } from "./jsrsasign.adapter";
import { CertificatesStore } from './certificates.store';
import { FileSignatureUpdater } from './file-signature.updater'; 
import { IAdapter } from './adapter.inteface';

export class SimpleCryptoProviderExtension implements ICryptoProvider, IInitializable {

  private _cryptoAdapter: IAdapter; 
  private _fileSignatureUpdater: FileSignatureUpdater;

  initialize(injectionSource: InjectionSource): void {
    this._fileSignatureUpdater = new FileSignatureUpdater(injectionSource.modifierProvider);
    this._cryptoAdapter = new JsRsaSignAdapter(new CertificatesStore());
    console.debug("SimpleCryptoProviderExtension has been initialized successfully");
  }

  canProcessAlgorithms(publicKeyOid: string): boolean {
    return JsRsaSignAdapter.signatureAlgorithm == publicKeyOid;
  }

  canProcessSignature(signatureFile: ArrayBuffer): boolean {
    return this._cryptoAdapter.canProcessSignature(signatureFile);
  }

  sign(documentId: string, actualFile: IFile, arrayBuffer: ArrayBuffer, certificate: ICertificate, signatureRequestIds: string[]): Observable<string> {
    const signature = from(this.signAction(documentId, actualFile, arrayBuffer, certificate, signatureRequestIds)) as Observable<string>;
    return signature;
  }

  verify(file: ArrayBuffer, sign: ArrayBuffer, signatureRequest: ISignatureRequest): Observable<ISignatureVerificationResult> {
    return of(this._cryptoAdapter.verify(file, sign, signatureRequest.lastSignCadesType)).pipe(tap(verificationResult => console.debug("Verification result: ",  verificationResult)));
  }

  getCertificates(): Observable<ICertificate[]> {
    return of(this._cryptoAdapter.getCertificates());
  }

  verifyImportedSignature(file: ArrayBuffer, sign: ArrayBuffer): Observable<IImportedSignatureVerificationResult> {
    return of(this._cryptoAdapter.verifyImportedSignature(file, sign));
  }

  private async signAction(documentId: string, actualFile: IFile, arrayBuffer: ArrayBuffer, certificate: ICertificate, signatureRequestIds: string[]): Promise<string> {
    const signature = this._cryptoAdapter.sign(arrayBuffer, certificate);
    await firstValueFrom(this._fileSignatureUpdater.setSignToObjectFile(documentId, actualFile, signature, certificate.publicKeyOid, signatureRequestIds));
    return signature;
  }
}