import { Component, inject, OnDestroy, OnInit } from '@angular/core';
import { MatFormFieldModule } from "@angular/material/form-field";
import { MatInputModule } from "@angular/material/input";
import { SettingsFeatureContext } from '@pilotdev/pilot-web-sdk';
import { SETTINGS_FEATURE_CONTEXT } from '../../tokens';
import { CommonModule } from '@angular/common';
import { FormBuilder, FormControl, FormGroup, ReactiveFormsModule, Validators } from '@angular/forms';
import { Subject, takeUntil } from 'rxjs';

interface ISampleFormData {
  firstName: string;
  lastName: string;
}

interface ISampleForm {
  firstName: FormControl<string>;
  lastName: FormControl<string>;
}

@Component({
  selector: 'app-sample-settings-feature-view',
  standalone: true,
  imports: [
    MatFormFieldModule,
    MatInputModule,
    CommonModule,
    ReactiveFormsModule
  ],
  templateUrl: './sample-settings-feature-view.component.html',
  styleUrl: './sample-settings-feature-view.component.scss'
})
export class SampleSettingsFeatureViewComponent implements OnInit, OnDestroy {
  private readonly _destroy$ = new Subject<void>();
  private readonly _fb: FormBuilder;
  private readonly _context: SettingsFeatureContext;

  form!: FormGroup<ISampleForm>;

  constructor() {
    this._fb = inject(FormBuilder);
    this._context = inject(SETTINGS_FEATURE_CONTEXT);
  }

  ngOnInit(): void {
    this.form = this._fb.nonNullable.group({
      firstName: ['', Validators.required],
      lastName: ['', Validators.required],
    });

    this.init();

    this.serializeOnChanges();
    this.setValidityOnChanges();
  }

  ngOnDestroy(): void {
    this._destroy$.next();
    this._destroy$.complete();
  }

  private init(): void {
    const valueStr = this._context.valueProvider.getValue();
    const value = JSON.parse(valueStr.length > 0 ? valueStr : '{}') as Partial<ISampleFormData>;

    if (value)
      this.form.patchValue(value);
  }

  private serializeOnChanges(): void {
    this.form.valueChanges.pipe(takeUntil(this._destroy$)).subscribe(() => {
      const strValue = JSON.stringify(this.form.value);
      this._context.valueProvider.setValue(strValue);
    });
  }

  private setValidityOnChanges(): void {
    this.form.statusChanges.pipe(takeUntil(this._destroy$)).subscribe((value) => {
      this._context.isValid = value === "VALID";
    });
  }
}
