/**
 * Copyright (c) 2024 ASCON-Design Systems LLC. All rights reserved.
 * This sample is licensed under the MIT License.
 */
import { ApplicationRef, ComponentRef, createComponent, Injectable, NgZone } from '@angular/core';
import { createApplication } from '@angular/platform-browser';

import { IDisposable, IOpenspaceView, SettingsFeatureContext } from '@pilotdev/pilot-web-sdk';
import { COMMON_SETTING_VIEW_ID } from '../../const';
import { SampleSettingsFeatureViewComponent } from '../../components/sample-settings-feature-view/sample-settings-feature-view.component';
import { SETTINGS_FEATURE_CONTEXT } from '../../tokens';

@Injectable({ providedIn: 'root' })
export class SampleSettingsFeatureView implements IOpenspaceView<SettingsFeatureContext>, IDisposable
{
  private _rootElement: HTMLElement | undefined;
  private _appRef: ApplicationRef | undefined;
  private _componentRef: ComponentRef<SampleSettingsFeatureViewComponent> | undefined;

  getViewId(): string {
    return COMMON_SETTING_VIEW_ID;
  }

  getView(context: SettingsFeatureContext): HTMLElement | undefined {
    if (!this._rootElement) {
      this._rootElement = document.createElement('div');
    }

    if (!this._appRef) {
      createApplication({
        providers: [
          { provide: SETTINGS_FEATURE_CONTEXT, useValue: context }
        ],
      }).then((appRef: ApplicationRef) => {
        this._appRef = appRef;

        const zone = appRef.injector.get(NgZone);
        zone.run(() => {
          this._componentRef = createComponent(SampleSettingsFeatureViewComponent, {
            environmentInjector: appRef.injector,
            hostElement: this._rootElement,
          });

          appRef.attachView(this._componentRef.hostView);
        });
      });
    }

    return this._rootElement;
  }

  dispose(): void {
    if (this._componentRef) {
      this._componentRef.destroy();
      this._componentRef = undefined;
    }

    if (this._appRef) {
      this._appRef.destroy();
      this._appRef = undefined;
    }

    if (this._rootElement) {
      this._rootElement = undefined;
    }
  }
}
