import {
  IMenu,
  IMenuBuilder,
  ObjectsViewContext,
  IRepositoryEvents,
  IObjectsRepository,
  IInitializable,
  InjectionSource,
} from '@pilotdev/pilot-web-sdk';
import { Subscription } from 'rxjs';
import { LocalStorageService } from './local-storage.service';
import { LocalStorageKeyGenerator } from './localstorage-key.generator';

export class SubscriptionsSample extends IMenu<ObjectsViewContext> implements IInitializable {
  private readonly EXTENTION_NAME = 'subscriptions_sample';
  private readonly STORAGE_KEY = 'subscriptions';
  private readonly SUBSCRIBE_ITEM_NAME = 'SubscribeItemName';
  private readonly UNSUBSCRIBE_ITEM_NAME = 'UnsubscribeItemName';

  private readonly _subscriptions: Map<string, Subscription> = new Map<string, Subscription>();
  private _localStorageService!: LocalStorageService;
  private _repositoryEvents!: IRepositoryEvents;
  private _objectsRepository!: IObjectsRepository;

  constructor() {
    super();
  }

  initialize(injectionSource: InjectionSource): void {
    this._repositoryEvents = injectionSource.repositoryEvents;
    this._objectsRepository = injectionSource.objectsRepository;

    this._localStorageService = new LocalStorageService(new LocalStorageKeyGenerator(this._objectsRepository, this.EXTENTION_NAME));

    for (const id of this.getSavedIds()) {
      this.subscribeObject(id);
    }
  }

  build(builder: IMenuBuilder, context: ObjectsViewContext): void {
    if (context.selectedObjects.length != 1)
      return;

    const indexItemName = 'accessRights';
    const insertIndex = builder.itemNames.indexOf(indexItemName) + 1;

    const id = context.selectedObjects[0].id;
    if (this._subscriptions.has(id)) {
      builder.addItem(this.UNSUBSCRIBE_ITEM_NAME, insertIndex).withHeader('Отписаться от изменений');
    } else {
      builder.addItem(this.SUBSCRIBE_ITEM_NAME, insertIndex).withHeader('Подписаться на изменения');
    }
  }

  onMenuItemClick(name: string, context: ObjectsViewContext): void {
    const id = context.selectedObjects[0].id;
    if (name === this.UNSUBSCRIBE_ITEM_NAME) {
      this.unsubscribeObject(id);
      this.saveToStorage();
      return;
    }

    this.subscribeObject(id);
    this.saveToStorage();
  }

  private unsubscribeObject(id: string): void {
    this._subscriptions.get(id)?.unsubscribe();
    this._subscriptions.delete(id);
  }

  private subscribeObject(id: string): void {
    const sub = this._repositoryEvents.subscribeObjects([ id ]).subscribe(x => {
      console.log(`элемент \`${ x.title }\` с идентификатором ${ x.id } изменился`);
    });
    this._subscriptions.set(id, sub);
  }

  private saveToStorage(): void {
    let keys = Array.from( this._subscriptions.keys() );
    this._localStorageService.setItem(this.STORAGE_KEY, JSON.stringify(keys));
  }

  private getSavedIds(): string[] {
    const json = this._localStorageService.getItem(this.STORAGE_KEY);
    if (!!json) 
      return JSON.parse(json);
    
    return [];
  }
}
