import { IFile } from '@pilotdev/pilot-web-sdk';
import { useMemo, useState } from 'react';
import { FileData, SortColumn, SortDirection } from '../../lib/models';
import { handleSort } from '../../lib/helpers';
import { FileTable } from './FileTable';
import styles from './FileTableView.module.css';
import FileInput from '../FileInput';

export interface FileTableViewProps {
  files: IFile[];
  onFileAdded: (file: File | undefined) => void;
  onDeleteAllFiles: () => void;
}

export const FileTableView = (props: FileTableViewProps) => {
  const [column, setColumn] = useState<SortColumn>(SortColumn.Name);
  const [direction, setDirection] = useState<SortDirection>(SortDirection.Asc);

  const filesData = useMemo<FileData[]>(
    () =>
      props.files.map(f => ({
        id: f.body.id,
        name: f.name,
        created: f.body.created,
        modified: f.body.modified,
        accessed: f.body.accessed,
        size: f.body.size,
      })),
    [props.files]
  );

  const sortedFiles = useMemo(() => {
    return handleSort(filesData, column, direction);
  }, [filesData, column, direction]);

  const onFileSelected = (file: File | undefined) => {
    props.onFileAdded(file);
  };

  const deleteAllFiles = () => {
    props.onDeleteAllFiles();
  }

  return (
    <div className={styles['view-container']}>
      <div className={styles.toolbar}>
        <select value={column} onChange={e => setColumn(e.target.value as SortColumn)}>
          <option value={SortColumn.Name}>Name</option>
          <option value={SortColumn.Created}>Created</option>
          <option value={SortColumn.Modified}>Modified</option>
          <option value={SortColumn.Accessed}>Accessed</option>
          <option value={SortColumn.Size}>Size</option>
        </select>
        <select value={direction} onChange={e => setDirection(e.target.value as SortDirection)}>
          <option value={SortDirection.Asc}>Ascending</option>
          <option value={SortDirection.Desc}>Descending</option>
        </select>
        <FileInput onFileSelected={onFileSelected} />
        <button onClick={deleteAllFiles}>Delete all files</button>
      </div>
      <FileTable files={sortedFiles} />
    </div>
  );
};
