﻿/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Linq;
using System.Windows;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using Ascon.Pilot.SDK;
using Ascon.Pilot.Bim.SDK.Model;
using Ascon.Pilot.Bim.SDK.Search;
using Ascon.Pilot.Bim.SDK.ModelTab;

namespace Ascon.Pilot.Bim.SDK.BottomPanelTabSample
{
    [Export(typeof(IDataPlugin))]
    public class BottomPanelTabSample : IDataPlugin
    {
        public const string TITLE = "Bottom panel tab sample";

        private readonly ITabsManager _tabsManager;
        private readonly IModelManager _modelManager;
        private readonly IModelSearchManager _modelSearchManager;
        private readonly IObjectsRepository _objectsRepository;
        private readonly Dictionary<Guid, BottomTabControlViewModel> _tabViewModels = new Dictionary<Guid, BottomTabControlViewModel>();

        [ImportingConstructor]
        public BottomPanelTabSample(IPilotServiceProvider pilotServiceProvider, IObjectsRepository objectsRepository)
        {
            _tabsManager = pilotServiceProvider.GetServices<ITabsManager>().First();
            _modelManager = pilotServiceProvider.GetServices<IModelManager>().First();
            _modelSearchManager = pilotServiceProvider.GetServices<IModelSearchManager>().First();
            _objectsRepository = objectsRepository;
            var openedTabs = _tabsManager.GetTabs().OfType<IModelTab>().ToList();
            InitOpenedTabsBottomPanel(openedTabs);
            Subscribe();
        }

        private void InitOpenedTabsBottomPanel(List<IModelTab> openedTabs)
        {
            openedTabs.ForEach(AddBottomTab);
        }

        private void AddBottomTab(IModelTab modelTab)
        {
            var bottomPanelManager = modelTab.GetBottomPanelManager();

            Application.Current.Dispatcher.Invoke(() =>
            {
                var viewModel = new BottomTabControlViewModel(_modelSearchManager, _objectsRepository);
                var bottomTab = bottomPanelManager.AddBottomTab(0, TITLE, new BottomTabControl { DataContext = viewModel });
                viewModel.BottomTab = bottomTab;

                if (modelTab.ModelViewer != null)
                {
                    bottomTab.IsVisible = true;
                    viewModel.OnModelLoaded(modelTab.ModelViewer);
                }
                else
                {
                    bottomTab.IsVisible = false;
                }

                _tabViewModels.Add(modelTab.Id, viewModel);
            });

            modelTab.Disposed += (_, __) => OnModelTabDisposed(modelTab.Id);
        }

        private void Subscribe()
        {
            _tabsManager.TabOpened += OnTabOpened;
            _modelManager.ModelLoaded += OnModelLoaded;
            _modelManager.ModelClosed += OnModelClosed;
        }

        private void OnTabOpened(object sender, TabEventArgs e)
        {
            if (e.Tab is IModelTab modelTab)
                AddBottomTab(modelTab);
        }
        
        private void OnModelLoaded(object sender, ModelEventArgs e)
        {
            _tabViewModels[e.Viewer.TabId].OnModelLoaded(e.Viewer);
        }

        private void OnModelClosed(object sender, ModelEventArgs e)
        {
            _tabViewModels[e.Viewer.TabId].OnModelClosed();
        }

        public void OnModelTabDisposed(Guid id)
        {
            _tabViewModels[id] = null;
            _tabViewModels.Remove(id);
        }
    }
}
