﻿using System;
using System.Collections.Generic;
using Ascon.Pilot.SDK;
using System.ComponentModel.Composition;
using System.Linq;
using Ascon.Pilot.Bim.SDK.CollisionJournal;
using Ascon.Pilot.Bim.SDK.ModelTab;
using Ascon.Pilot.Bim.SDK.Model;
using Ascon.Pilot.Bim.SDK.ModelStorage;

namespace Ascon.Pilot.Bim.SDK.CollisionJournalSample
{
    [Export(typeof(IDataPlugin))]
    public class CollisionJournalSample : IDataPlugin
    {
        private readonly ITabsManager _tabsManager;
        private readonly IModelManager _modelManager;
        private readonly ICollisionJournalProvider _collisionJournalProvider;
        private readonly IModelStorageProvider _modelStorageProvider;
        private readonly IObjectsRepository _objectsRepository;

        private readonly Dictionary<Guid, ToolbarItemController> _toolbarItemControllers = new Dictionary<Guid, ToolbarItemController>();

        [ImportingConstructor]
        public CollisionJournalSample(IPilotServiceProvider pilotServiceProvider, IObjectsRepository objectsRepository)
        {
            _objectsRepository = objectsRepository;
            _tabsManager = pilotServiceProvider.GetServices<ITabsManager>().First();
            _modelManager = pilotServiceProvider.GetServices<IModelManager>().First();
            _collisionJournalProvider = pilotServiceProvider.GetServices<ICollisionJournalProvider>().First();
            _modelStorageProvider = pilotServiceProvider.GetServices<IModelStorageProvider>().First();
            var openedTabs = _tabsManager.GetTabs().OfType<IModelTab>().ToList();
            InitOpenedTabsToolbar(openedTabs);
            Subscribe();
        }

        private void InitOpenedTabsToolbar(List<IModelTab> openedTabs)
        {
            openedTabs.ForEach(AddToolbarItemController);
        }

        private void AddToolbarItemController(IModelTab modelTab)
        {
            var toolbarItemController = new ToolbarItemController(modelTab, _objectsRepository, _collisionJournalProvider, _modelStorageProvider);
            toolbarItemController.AddToolbarItem();
            _toolbarItemControllers.Add(modelTab.Id, toolbarItemController);
            modelTab.Disposed += (_, __) => OnModelTabDisposed(modelTab.Id);
        }

        private void Subscribe()
        {
            _tabsManager.TabOpened += OnTabOpened;
            _modelManager.ModelLoaded += OnModelLoaded;
            _modelManager.ModelClosed += OnModelClosed;
        }

        private void OnTabOpened(object sender, TabEventArgs e)
        {
            if (e.Tab is IModelTab modelTab)
                AddToolbarItemController(modelTab);
        }

        private void OnModelLoaded(object sender, ModelEventArgs e)
        {
            foreach (var controller in _toolbarItemControllers)
                controller.Value.OnModelLoaded(e.Viewer);
        }

        private void OnModelClosed(object sender, ModelEventArgs e)
        {
            foreach (var controller in _toolbarItemControllers)
                controller.Value.OnModelClosed(e.Viewer);
        }

        public void OnModelTabDisposed(Guid id)
        {
            _toolbarItemControllers[id].Dispose();
            _toolbarItemControllers[id] = null;
            _toolbarItemControllers.Remove(id);
        }
    }
}
