﻿/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Linq;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using Ascon.Pilot.SDK;
using Ascon.Pilot.Bim.SDK.Model;
using Ascon.Pilot.Bim.SDK.ModelTab;
using Ascon.Pilot.Bim.SDK.ImportSearchSetsSample.Import;

namespace Ascon.Pilot.Bim.SDK.ImportSearchSetsSample
{
    [Export(typeof(IDataPlugin))]
    public class ImportSearchSetsSample : IDataPlugin
    {
        private readonly IObjectsRepository _objectsRepository;
        private readonly IObjectModifier _objectModifier;
        private readonly ITabsManager _tabsManager;
        private readonly IModelManager _modelManager;
        
        private readonly Dictionary<Guid, ToolbarItemsController> _toolbarItemControllers = new Dictionary<Guid, ToolbarItemsController>();

        [ImportingConstructor]
        public ImportSearchSetsSample(IPilotServiceProvider pilotServiceProvider, IObjectsRepository repository, IObjectModifier modifier)
        {
            _objectsRepository = repository;
            _objectModifier = modifier;
            _tabsManager = pilotServiceProvider.GetServices<ITabsManager>().First();
            _modelManager = pilotServiceProvider.GetServices<IModelManager>().First();
            var openedTabs = _tabsManager.GetTabs().OfType<IModelTab>().ToList();
            InitOpenedTabsToolbar(openedTabs);
            Subscribe();
        }

        private void InitOpenedTabsToolbar(List<IModelTab> openedTabs)
        {
            openedTabs.ForEach(AddToolbarItemsController);
        }

        private void AddToolbarItemsController(IModelTab modelTab)
        {
            var toolbarItemController = new ToolbarItemsController(modelTab);
            toolbarItemController.AddToolbarItems();
            _toolbarItemControllers.Add(modelTab.Id, toolbarItemController);
            modelTab.Disposed += (_, __) => OnModelTabDisposed(modelTab.Id);
        }

        private void Subscribe()
        {
            _tabsManager.TabOpened += OnTabOpened;
            _modelManager.ModelLoaded += OnModelLoaded;
            _modelManager.ModelClosed += OnModelClosed;
        }

        private void OnTabOpened(object sender, TabEventArgs e)
        {
            if (e.Tab is IModelTab modelTab)
                AddToolbarItemsController(modelTab);
        }

        private void OnModelLoaded(object sender, ModelEventArgs e)
        {
             if (_toolbarItemControllers.TryGetValue(e.Viewer.TabId, out var controller))
                controller.OnModelLoaded(e.Viewer, new ImportService(e.Viewer.ModelId, _objectsRepository, _objectModifier));
        }

        private void OnModelClosed(object sender, ModelEventArgs e)
        {
            if (_toolbarItemControllers.TryGetValue(e.Viewer.TabId, out var controller))
                controller.OnModelClosed(e.Viewer);
        }

        public void OnModelTabDisposed(Guid id)
        {
            _toolbarItemControllers[id].Dispose();
            _toolbarItemControllers[id] = null;
            _toolbarItemControllers.Remove(id);
        }
    }
}
