﻿/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.Diagnostics;
using System.IO;
using System.Reflection;
using Ascon.Pilot.Bim.SDK.RengaComAdapterSample;
using Ascon.Pilot.Bim.SDK.RengaModuleSample.Properties;
using Ascon.Pilot.SDK;
using Ascon.Pilot.SDK.Menu;

namespace Ascon.Pilot.Bim.SDK.RengaModuleSample
{
    [Export(typeof(IDataPlugin))]
    public class RengaModule : IDataPlugin, IMenu<IModelElementId>, IObserver<IDataObject>
    {
        private readonly IObjectsRepository _objectsRepository;
        private readonly IClientInfo _clientInfo;
        private readonly IPilotDialogService _dialogService;

        private const string SHOW_OBJECT_IN_RENGA_COMMAND_NAME = "miShowObjectInRengaCommand";
        private const string COPY_LINK_TO_OBJECT_IN_RENGA_COMMAND_NAME = "miCopyLinkToObjectInRengaCommand";
        private const string RENGA_ICON_NAME = "Renga_logo.svg";
        private const string MODEL_PART_IFC_APPLICATION_ID_ATTR_NAME = "IFCAPPLICATION_ID";
        private const string MODEL_PART_FILE_NAME_ATTR_NAME = "FILE_NAME";
        private const string RENGA_APP_IFC_NAME = "RENGA";

        private IMenuItemBuilder _showObjectInRengaMenuItemBuilder;
        private IMenuItemBuilder _copyLinkToObjectInRengaMenuItemBuilder;
        private IDisposable _modelPartSubscription;

        [ImportingConstructor]
        public RengaModule(IObjectsRepository repository, IClientInfo clientInfo, IPilotServiceProvider serviceProvider, IPilotDialogService dialogService)
        {
            _objectsRepository = repository;
            _clientInfo = clientInfo;
            _dialogService = dialogService;
            serviceProvider.Register<IMenu<IModelElementId>>(this);
            
            //запустим RengaProtocolHandler (приложение само зарегистрируется, если нужно)
            StartProtocolHandler();
        }

        private void StartProtocolHandler()
        {
            //считаем, что RengaProtocolHandler.exe лежит в корневой папке плагина
            var executingAssemblyFilename = Assembly.GetExecutingAssembly().Location;
            var directory = Directory.GetParent(executingAssemblyFilename);
            var exeFilename = Path.Combine(directory.FullName, "RengaProtocolHandler.exe");
            if (!File.Exists(exeFilename))
            {
                //todo log
                return;
            }

            var p = new Process
            {
                StartInfo =
                {
                    UseShellExecute = false, 
                    FileName = exeFilename, 
                    Arguments = "--init"

                }
            };
            p.Start();
        }

        public void Build(IMenuBuilder builder, IModelElementId elemendId)
        {
            if (elemendId.ModelPartId.Equals(Guid.Empty))
                return;

            _modelPartSubscription?.Dispose();

            var iconBytes = IconLoader.GetIcon(RENGA_ICON_NAME);

            builder.AddSeparator(builder.Count - 1);

            _showObjectInRengaMenuItemBuilder = builder.AddItem(SHOW_OBJECT_IN_RENGA_COMMAND_NAME, builder.Count - 1)
                .WithHeader(Resources.ShowObjectInRengaCommand)
                .WithIcon(iconBytes)
                .WithIsEnabled(false);

            _copyLinkToObjectInRengaMenuItemBuilder = builder.AddItem(COPY_LINK_TO_OBJECT_IN_RENGA_COMMAND_NAME, builder.Count - 1)
                .WithHeader(Resources.CopyLinkToObjectInRengaCommand)
                .WithIcon(iconBytes)
                .WithIsEnabled(false);

            var observable = _objectsRepository.SubscribeObjects(new[] { elemendId.ModelPartId });
            _modelPartSubscription = observable.Subscribe(this);
        }

        public void OnMenuItemClick(string name, IModelElementId elementId)
        {
            if (name == SHOW_OBJECT_IN_RENGA_COMMAND_NAME)
                ShowObjectInRenga(elementId.ElementId);
            else if (name == COPY_LINK_TO_OBJECT_IN_RENGA_COMMAND_NAME)
                UriHelper.CopyObjectUriToClipboard(elementId.ElementId, _clientInfo.ConnectionString);
        }

        private void ShowObjectInRenga(Guid id)
        {
            var result = Adapter.ShowObjectInRenga(id);

            if (!result)
                _dialogService.ShowBalloon(Resources.ShowObjectInRengaTitle, Resources.RengaInstanceWithSourceProjectNotFoundMessage, PilotBalloonIcon.Info);
        }

        private bool CanProvideCommands(IDictionary<string, object> attributes)
        {
            if (attributes.TryGetValue(MODEL_PART_IFC_APPLICATION_ID_ATTR_NAME, out var appId))
            {
                if (appId.ToString().ToUpper().Contains(RENGA_APP_IFC_NAME))
                    return true;
            }

            if (attributes.TryGetValue(MODEL_PART_FILE_NAME_ATTR_NAME, out var fileName))
            {
                if (fileName.ToString().ToUpper().Contains(RENGA_APP_IFC_NAME))
                    return true;
            }

            return false;
        }

        public void OnNext(IDataObject value)
        {
            if (CanProvideCommands(value.Attributes))
            {
                _showObjectInRengaMenuItemBuilder.WithIsEnabled(true);
                _copyLinkToObjectInRengaMenuItemBuilder.WithIsEnabled(true);
            }

            _modelPartSubscription.Dispose();
            _modelPartSubscription = null;
        }

        public void OnError(Exception error)
        {
        }

        public void OnCompleted()
        {
        }
    }
}