﻿/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Text.RegularExpressions;
using Ascon.Pilot.Bim.SDK.RengaComAdapterSample;
using Microsoft.Win32;

namespace Ascon.Pilot.Bim.SDK.RengaProtocolHandlerSample
{
    class Program
    {
        static void Main(string[] args)
        {
            if (args.Length > 0)
            {
                var protocol = "rengauri";
                var command = args[0];
                switch (command)
                {
                    case "--init" :
                        if (IsProtocolRegistered(protocol))
                            return;
                        RegisterProtocol(protocol, $"URL:Renga Protocol");
                        Console.WriteLine($"Protocol '{protocol}' registered successfully.");
                        break;
                    case "--register":
                        RegisterProtocol(protocol, $"URL:Renga Protocol");
                        Console.WriteLine($"Protocol '{protocol}' registered successfully.");
                        break;
                    case "--unregister":
                        UnregisterProtocol(protocol);
                        Console.WriteLine($"Protocol '{protocol}' unregistered successfully.");
                        break;
                    default :
                        if (command.Contains(protocol))
                        {
                            var id = HandleProtocol(command);
                            Console.WriteLine($"Run application with id = {id}");
                            ShowObjectInRenga(id);
                        }
                        else
                        {
                            Console.WriteLine("Wrong parameters");
                            Console.WriteLine(string.Join(", ", args));
                        }
                        break;
                }
            }
            else
            {
                Console.WriteLine("No params set.");
            }

#if DEBUG
            Console.Write("Press any key to exit...");
            Console.ReadKey();
#endif
        }

        private static void RegisterProtocol(string protocol, string description)
        {
            var filename = System.Reflection.Assembly.GetExecutingAssembly().Location;
            using (var currentUserKey = RegistryKey.OpenBaseKey(RegistryHive.CurrentUser, RegistryView.Default))
            using (var subKey = currentUserKey.OpenSubKey(@"SOFTWARE\classes", true))
            {
                var registryKey = subKey.CreateSubKey(protocol);
                registryKey.SetValue("", description);
                registryKey.DeleteValue("URL Protocol", false);
                registryKey.SetValue("URL Protocol", "");
                var shellKey = registryKey.CreateSubKey("shell");
                var openKey = shellKey.CreateSubKey("open");
                var commandKey = openKey.CreateSubKey("command");
                commandKey.SetValue("", "\"" + filename + "\"" + " \"%1\"");
            }
        }

        private static void UnregisterProtocol(string protocol)
        {
            using (var currentUserKey = RegistryKey.OpenBaseKey(RegistryHive.CurrentUser, RegistryView.Default))
            using (var subKey = currentUserKey.OpenSubKey(@"SOFTWARE\classes", true))
            {
                subKey?.DeleteSubKeyTree(protocol, false);
            }
        }

        private static bool IsProtocolRegistered(string protocol)
        {
            using (var currentUserKey = RegistryKey.OpenBaseKey(RegistryHive.CurrentUser, RegistryView.Default))
            using (var subKey = currentUserKey.OpenSubKey($"SOFTWARE\\classes\\{protocol}\\shell\\open\\command", false))
            {
                if (subKey == null)
                    return false;

                var value = subKey.GetValue("") as string;
                var filename = System.Reflection.Assembly.GetExecutingAssembly().Location;
                return value == $"\"{filename}\" \"%1\"";
            }
        }

        // handle url rengauri://50845F78-DECC-4274-BB9B-B7FE4B76E270
        private static string HandleProtocol(string url)
        {
            var args = Regex.Split(url, "/");
            if (args.Length >= 3)
                return args[2];

            return string.Empty;
        }

        private static void ShowObjectInRenga(string guid)
        {
            if (string.IsNullOrEmpty(guid))
                return;

            if (Guid.TryParse(guid, out var id))
                Adapter.ShowObjectInRenga(id);
        }
    }
}
