﻿/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Linq;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using Ascon.Pilot.SDK;
using Ascon.Pilot.Bim.SDK.Model;
using Ascon.Pilot.Bim.SDK.ModelTab;

namespace Ascon.Pilot.Bim.SDK.ToolbarSample
{
    [Export(typeof(IDataPlugin))]
    public class ToolbarSample : IDataPlugin
    {
        private readonly ITabsManager _tabsManager;
        private readonly IModelManager _modelManager;

        private readonly Dictionary<Guid, ToolbarItemsController> _toolbarItemControllers = new Dictionary<Guid, ToolbarItemsController>();

        [ImportingConstructor]
        public ToolbarSample(IPilotServiceProvider pilotServiceProvider)
        {
            _tabsManager = pilotServiceProvider.GetServices<ITabsManager>().First();
            _modelManager = pilotServiceProvider.GetServices<IModelManager>().First();
            var openedTabs = _tabsManager.GetTabs().OfType<IModelTab>().ToList();
            InitOpenedTabsToolbar(openedTabs);
            Subscribe();
        }

        private void InitOpenedTabsToolbar(List<IModelTab> openedTabs)
        {
            openedTabs.ForEach(AddToolbarItemsController);
        }

        private void AddToolbarItemsController(IModelTab modelTab)
        {
            var toolbarItemController = new ToolbarItemsController(modelTab);
            toolbarItemController.AddToolbarItems();
            _toolbarItemControllers.Add(modelTab.Id, toolbarItemController);
            modelTab.Disposed += (_, __) => OnModelTabDisposed(modelTab.Id);
        }

        private void Subscribe()
        {
            _tabsManager.TabOpened += OnTabOpened;
            _modelManager.ModelLoaded += OnModelLoaded;
            _modelManager.ModelClosed += OnModelClosed;
        }

        private void OnTabOpened(object sender, TabEventArgs e)
        {
            if (e.Tab is IModelTab modelTab)
                AddToolbarItemsController(modelTab);
        }

        private void OnModelLoaded(object sender, ModelEventArgs e)
        {
            foreach (var controller in _toolbarItemControllers)
                controller.Value.OnModelLoaded(e.Viewer);
        }

        private void OnModelClosed(object sender, ModelEventArgs e)
        {
            foreach (var controller in _toolbarItemControllers)
                controller.Value.OnModelClosed(e.Viewer);
        }

        public void OnModelTabDisposed(Guid id)
        {
            _toolbarItemControllers[id].Dispose();
            _toolbarItemControllers[id] = null;
            _toolbarItemControllers.Remove(id);
        }
    }
}