/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.ComponentModel.Composition;
using System.IO;
using System.Linq;
using System.Windows;

namespace Ascon.Pilot.SDK.AutoimportSample
{
    [Export(typeof(IAutoimportHandler))]
    public class AutoimportHandler : IAutoimportHandler
    {
        private static Guid _parentId = SystemObjectIds.RootObjectId;

        private readonly IObjectModifier _modifier;
        private readonly IPilotDialogService _dialogService;
        
        [ImportingConstructor]
        public AutoimportHandler(IObjectModifier modifier, IPilotDialogService dialogService)
        {
            _modifier = modifier;
            _dialogService = dialogService;
        }

        public bool Handle(string filePath, string sourceFilePath, AutoimportSource autoimportSource)
        {
            try
            {
                var selection = _dialogService.ShowDocumentsSelectorDialogAndNavigate(_parentId).ToList();
                if (selection.Count != 1)
                    return false;
                var document = selection.First();
                if (!document.Type.HasFiles)
                    MessageBox.Show("Error", "Selected element can not have files", MessageBoxButton.OK, MessageBoxImage.Error);

                _parentId = document.ParentId;

                var message = "Auto-imported from " + Localize(autoimportSource);
                _modifier
                    .Edit(document)
                    .CreateFileSnapshot(message)
                    .AddFile(filePath);
                _modifier.Apply();
                return true;
            }
            finally
            {
                File.Delete(filePath);
            }
        }

        private string Localize(AutoimportSource autoimportSource)
        {
            switch (autoimportSource)
            {
                case AutoimportSource.Unknown:
                    return "Unknown";
                case AutoimportSource.PilotXps:
                    return "Pilot XPS printer";
                case AutoimportSource.UserFolder:
                    return "user auto-import directory";
                default:
                    throw new NotSupportedException();
            }
        }
    }
}
