/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Diagnostics;
using System.IO.Ports;
using System.Linq;
using System.Windows;
using System.Windows.Media;
using System.ComponentModel.Composition;
using Ascon.Pilot.SDK.Menu;
using Ascon.Pilot.Theme.ColorScheme;

namespace Ascon.Pilot.SDK.BarcodeScannerSample
{
    [Export(typeof(IMenu<MainViewContext>))]
    public class BarcodeScannerReader : IMenu<MainViewContext>, IHandle<UnloadedEventArgs>
    {
        private const string SERVICE_BARCODE_MENU = "ServiceBarcodeMenu";
        private const string PILOT_URI = "piloturi://";

        private SerialPort _port;
        private BarcodeScannerSettingsView _settingsView;
        private BarcodeScannerSettingsModel _model;

        [ImportingConstructor]
        public BarcodeScannerReader(IEventAggregator eventAggregator, IPilotDialogService dialogService)
        {
            var convertFromString = ColorConverter.ConvertFromString(dialogService.AccentColor);
            if (convertFromString != null)
            {
                var color = (Color)convertFromString;
                ColorScheme.Initialize(color, dialogService.Theme);
            }
            var portName = CheckSavedPort();
            if (portName == null)
                return;
            _port = new SerialPort(portName);
            if (!_port.IsOpen)
                SafePortOpenner.OpenPort(_port);
            _port.DataReceived += PortOnDataReceived;
            eventAggregator.Subscribe(this);
        }

        public void Build(IMenuBuilder builder, MainViewContext context)
        {
            var menuItem = builder.ItemNames.First();
            builder.GetItem(menuItem).AddItem(SERVICE_BARCODE_MENU, 0).WithHeader(Properties.Resources.txtMenuItem);
        }

        public void OnMenuItemClick(string name, MainViewContext context)
        {
            if (name == SERVICE_BARCODE_MENU)
            {
                if (_port != null)
                {
                    _port.Close();
                    _port.DataReceived -= PortOnDataReceived;
                }
                _model = new BarcodeScannerSettingsModel();
                _settingsView = new BarcodeScannerSettingsView { DataContext = _model };
                _settingsView.Unloaded += SettingsViewOnUnloaded;
                _settingsView.ShowDialog();
            }
        }

        private string CheckSavedPort()
        {
            var portName = Properties.Settings.Default.Port;
            if (SerialPort.GetPortNames().Contains(portName))
                return portName;
            return SerialPort.GetPortNames().FirstOrDefault();
        }

        private void PortOnDataReceived(object sender, SerialDataReceivedEventArgs serialDataReceivedEventArgs)
        {
            var data = _port.ReadLine();
            Tuple<Guid,  long?, short?> res = null;
            res = DecodeDocumentLink(data.Trim());
            if (res == null)
                return;
            if (res.Item2 != null) 
                Process.Start(PILOT_URI + res.Item1 + "/" + res.Item2);
            if (res.Item3 != null)
                Process.Start(PILOT_URI + res.Item1 + "/" + res.Item3);
        }

        private Tuple<Guid,  long?, short?> DecodeDocumentLink(string base64)
        {
            try
            {
                Guid guid;
                if (Guid.TryParse(base64, out guid))
                    return null;
                
                base64 = base64.Replace("-", "/").Replace("_", "+");
                var byteArray = Convert.FromBase64String(base64);
                switch (byteArray.Length)
                {
                    case 18:
                        //поддержка старого кодирования
                        return new Tuple<Guid, long?, short?>(new Guid(byteArray.Take(16).ToArray()), null,
                            BitConverter.ToInt16(byteArray, 16));

                    case 24:
                        return new Tuple<Guid, long?, short?>(new Guid(byteArray.Take(16).ToArray()),
                            BitConverter.ToInt64(byteArray, 16), null);
                    default:
                        return null;
                }
            }
            catch
            {
                return null;
            }
        }

        private void SettingsViewOnUnloaded(object sender, RoutedEventArgs routedEventArgs)
        {
            _settingsView.Unloaded -= SettingsViewOnUnloaded;
            var portName = CheckSavedPort();
            if (portName == null)
                return;
            _port = new SerialPort(CheckSavedPort());
            _port.DataReceived += PortOnDataReceived;
            SafePortOpenner.OpenPort(_port);
        }

        public void Handle(UnloadedEventArgs message)
        {
            _port.Close();
        }
    }
}
