/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Globalization;
using System.IO.Ports;
using System.Linq;
using System.Threading;

namespace Ascon.Pilot.SDK.BarcodeScannerSample
{
    class BarcodeScannerSettingsModel : INotifyPropertyChanged
    {
        private SerialPort _currentComPort;
        private string _scanningResult;
        private CultureInfo _cultureInfo;

        public ObservableCollection<SerialPort> AvailableComPorts { get; set; }
        public event PropertyChangedEventHandler PropertyChanged;

        public SerialPort CurrentComPort
        {
            get { return _currentComPort; }
            set
            {
                if (_currentComPort != null)
                {
                    _currentComPort.DataReceived -= PortOnDataReceived;
                    _currentComPort.Close();
                }
                _currentComPort = value;
                if (value != null)
                {
                    _currentComPort.DataReceived += PortOnDataReceived;
                    SafePortOpenner.OpenPort(_currentComPort);
                }
                NotifyPropertyChanged("CurrentComPort");
                ScaningResult = "";
            }
        }

        public string ScaningResult
        {
            set
            {
                Thread.CurrentThread.CurrentUICulture = _cultureInfo;
                if (string.IsNullOrEmpty(value.Trim()))
                    _scanningResult = Properties.Resources.txtWaitingForScanning;
                else
                    _scanningResult = Properties.Resources.txtScanningResult + " " + value.Trim();
                NotifyPropertyChanged("ScaningResult");
            }
            get { return _scanningResult; }
        }

        public BarcodeScannerSettingsModel()
        {
            _cultureInfo = Thread.CurrentThread.CurrentUICulture;
            AvailableComPorts = new ObservableCollection<SerialPort>();
        }

        private void PortOnDataReceived(object sender, SerialDataReceivedEventArgs e)
        {
            ScaningResult = _currentComPort.ReadLine();
        }

        public void InitComPortsList()
        {
            foreach (var portName in SerialPort.GetPortNames())
                AvailableComPorts.Add(new SerialPort(portName));
            var savedPort = AvailableComPorts.FirstOrDefault(p => p.PortName.Equals(Properties.Settings.Default.Port));
            CurrentComPort = savedPort ?? AvailableComPorts.FirstOrDefault();
        }

        public void NotifyPropertyChanged(string name)
        {
            if (PropertyChanged != null)
                PropertyChanged(this, new PropertyChangedEventArgs(name));
        }

        public void SaveSettings()
        {
            Properties.Settings.Default.Port = CurrentComPort != null ? CurrentComPort.PortName : null;
            Properties.Settings.Default.Save();
        }

        public void DisposeResources()
        {
            if (CurrentComPort == null)
                return;
            CurrentComPort.DataReceived -= PortOnDataReceived;
            CurrentComPort.Close();
        }
    }
}
