﻿/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using Ascon.Pilot.SDK.Menu;
using System;
using System.ComponentModel.Composition;
using System.Collections.Generic;
using Ascon.Pilot.SDK.Common.WorkingDaysCalendar;
using System.Text;
using System.Linq;
using System.Threading.Tasks;
using Ascon.Pilot.SDK.Controls;
using System.Windows;

namespace Ascon.Pilot.SDK.CalendarSample
{
    [Export(typeof(IMenu<CalendarObjectsViewContext>))]
    public class CalendarSample : IMenu<CalendarObjectsViewContext>
    {
        private const string TODAY_WORKING_HOURS_COMMAND_NAME = "today_wh";
        private const string OBJECT_CREATORS_TODAY_WORKING_TIME_COMMAND_NAME = "obj_creator_working_time";
        private const string WEEKLY_WORKING_HOURS_COMMAND_NAME = "week_wh";
        private const string DATE_AFTER_TEN_WORKING_DAYS_COMMAND_NAME = "date_10";

        private IWorkingCalendarManager _calendarManager;
        private IWorkingCalendarService _workingCalendar;
        private IObjectsRepository _repository;
        private ObjectLoader _loader;
        private IPerson _currentPerson;

        [ImportingConstructor]
        public CalendarSample(IWorkingCalendarService workingCalendar, IWorkingCalendarManager calendarManager, IObjectsRepository repository)
        {
            _calendarManager = calendarManager;
            _workingCalendar = workingCalendar;
            _repository = repository;
            _loader = new ObjectLoader(repository);
            _currentPerson = _repository.GetCurrentPerson();
        }

        public void Build(IMenuBuilder builder, CalendarObjectsViewContext context)
        {
            builder.AddItem(TODAY_WORKING_HOURS_COMMAND_NAME, 0).WithHeader("My working hours today");
            builder.AddItem(WEEKLY_WORKING_HOURS_COMMAND_NAME, 1).WithHeader("My working hours for the week");
            builder.AddItem(DATE_AFTER_TEN_WORKING_DAYS_COMMAND_NAME, 2).WithHeader("The date after my 10 workind days is");
            if (context.SelectedObjects.Count() == 1)
                builder.AddItem(OBJECT_CREATORS_TODAY_WORKING_TIME_COMMAND_NAME, 3).WithHeader("Object creator's working time today");
        }

        public void OnMenuItemClick(string name, CalendarObjectsViewContext context)
        {

            Task.Run(async () =>
            {
                try
                {
                    if (name == TODAY_WORKING_HOURS_COMMAND_NAME)
                        await ShowTodayWorkingHoursAsync();

                    if (name == OBJECT_CREATORS_TODAY_WORKING_TIME_COMMAND_NAME)
                        await ShowObjectCreatorTodaysWorkingTimeAsync(context.SelectedObjects.First().Creator);

                    if (name == WEEKLY_WORKING_HOURS_COMMAND_NAME)
                        await ShowWorkingHoursForWeekAsync();

                    if (name == DATE_AFTER_TEN_WORKING_DAYS_COMMAND_NAME)
                        await ShowWorkingDateAsync(10);
                }
                catch (Exception ex)
                {
                    MessageBox.Show(ex.Message);
                }
            });
        }

        private async Task ShowWorkingDateAsync(int days)
        {
            var date = await _calendarManager.GetPersonWorkingDateAsync(DateTime.Now, days, _currentPerson.Id);
            MessageBox.Show($"My {days} working days will end on {date.ToString()}");
        }

        private async Task ShowWorkingHoursForWeekAsync()
        {
            var sb = new StringBuilder();
            sb.AppendLine("Working hours:");
            foreach (var day in GetDaysOfCurrentWeek())
            {
                var calendar = await _workingCalendar.GetPersonCalendarDayAsync(_currentPerson.Id, day);
                if (!calendar.WorkCalendars.Any())
                {
                    MessageBox.Show($"No working calendar found.");
                    return;
                }

                var eventName = await GetEventNameAsync(calendar);
                sb.AppendLine($"{day.DayOfWeek}: {GetWorkingHoursString(calendar.WorkingHours)} Event: {eventName}");
            }
            MessageBox.Show(sb.ToString());
        }

        private async Task ShowTodayWorkingHoursAsync()
        {
            var calendar = await _workingCalendar.GetPersonCalendarDayAsync(_currentPerson.Id, DateTime.Now);
            if (!calendar.WorkCalendars.Any())
            {
                MessageBox.Show($"No working calendar found.");
                return;
            }
            var calendarName = await GetCalendarNameAsync(calendar);
            var eventName = await GetEventNameAsync(calendar);

            MessageBox.Show($"Working hours today: {GetWorkingHoursString(calendar.WorkingHours)}\nCalendar: {calendarName}\nEvent: {eventName}");
        }

        private async Task<string> GetCalendarNameAsync(ICalendarDay calendarDay)
        {
            var obj = await _loader.Load(calendarDay.WorkCalendars.LastOrDefault().ObjectId);
            return obj.Attributes["name"].ToString();
        }

        private async Task<string> GetEventNameAsync(ICalendarDay calendarDay)
        {
            if (!calendarDay.WorkCalendarExceptions.Any())
                return "-";

            var mostRelevantEvent = calendarDay.WorkCalendarExceptions.Last();
            var obj = await _loader.Load(mostRelevantEvent.ObjectId);
            return obj.Attributes["name"].ToString();
        }

        private IList<DateTime> GetDaysOfCurrentWeek()
        {
            DateTime today = DateTime.Today;
            int daysToSubtract = ((int)today.DayOfWeek - (int)DayOfWeek.Monday + 7) % 7;
            DateTime startOfWeek = today.AddDays(-daysToSubtract);

            return Enumerable.Range(0, 7).Select(i => startOfWeek.AddDays(i)).ToList();
        }

        private string GetWorkingHoursString(IReadOnlyCollection<WorkingHours> wh)
        {
            if (wh == null || !wh.Any())
                return "-";
            var strWh = wh.Select(x => $"{x.From.Hour}:{x.From.Minute}-{x.To.Hour}:{x.To.Minute}");
            return string.Join(", ", strWh);
        }

        private async Task ShowObjectCreatorTodaysWorkingTimeAsync(IPerson creator)
        {
            var calendar = await _workingCalendar.GetPersonCalendarDayAsync(creator.Id, DateTime.Now);
            if (!calendar.WorkCalendars.Any())
            {
                MessageBox.Show($"No working calendar found.");
                return;
            }
            var calendarName = await GetCalendarNameAsync(calendar);
            var eventName = await GetEventNameAsync(calendar);


            MessageBox.Show($"{creator.Login} working time today: {GetWorkingTimeString(calendar.WorkingHours, creator.TimeZone)}\nCalendar: {calendarName}\nEvent: {eventName}");
        }

        private string GetWorkingTimeString(IReadOnlyCollection<WorkingHours> workingHours, string timeZone)
        {
            if (workingHours == null || !workingHours.Any())
                return "-";

            var result = string.Empty;
            var now = DateTime.Now;
            var sb = new StringBuilder();


            foreach (var wh in workingHours)
            {
                var fromShifted = TimeZoneHandler.ShiftToTimeZone(now.Date + wh.From.AsTimeSpan(), timeZone);
                var toShifted = TimeZoneHandler.ShiftToTimeZone(now.Date + wh.To.AsTimeSpan(), timeZone);
                result += $"{fromShifted} : {toShifted}\n";
            }

            return result;
        }

    }
}
