﻿/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.ComponentModel.Composition;
using System.Linq;
using System.Runtime.Remoting;
using System.Threading.Tasks;
using System.Windows;
using Ascon.Pilot.SDK.Data;

namespace Ascon.Pilot.SDK.ChatsSample
{
    [Export(typeof(INewTabPage))]
    [ExportMetadata("Name", "Chats Sample")]
    public class ChatsSample : INewTabPage, IObserver<IChatInfo>, IObserver<IChatMessage>
    {
        private readonly IMessagesRepository _repository;
        private readonly IObjectsRepository _objRepository;
        private readonly ITabServiceProvider _tabServiceProvider;
        private const string OPEN_CHATS_COMMAND = "OpenChatsCommand";
        private MessagesViewModel _viewModel;
        private List<IChatInfo> _loadedChats;

        [ImportingConstructor]
        public ChatsSample(IMessagesRepository repository, IObjectsRepository objRepository, ITabServiceProvider tabServiceProvider)
        {
            _repository = repository;
            _objRepository = objRepository;
            _tabServiceProvider = tabServiceProvider;
            _loadedChats = new List<IChatInfo>();
        }

        public void BuildNewTabPage(INewTabPageHost host)
        {
            host.AddButton("Chats Sample", OPEN_CHATS_COMMAND, "Open tab with chats sample", null);
        }

        public async void OnButtonClick(string name)
        {
            if (name != OPEN_CHATS_COMMAND)
                return;

            var title = _tabServiceProvider.GetActiveTabPageTitle();
            
            var view = await CreateChatsView();
            _tabServiceProvider.UpdateTabPageContent(title, "Chats Sample", view);
        }

        private async Task<MessagesView> CreateChatsView()
        {
            var chatInfos = await _repository.LoadChatsAsync(DateTime.Now, 100, false);
            _loadedChats = chatInfos.ToList();
            
            var list = new List<MessageViewModel>();
            foreach (var chatInfo in chatInfos)
            {
                var messages =
                    await _repository.LoadMessagesAsync(chatInfo.Chat.Id, DateTime.MinValue, DateTime.MaxValue, Int32.MaxValue);

                foreach (var chatMessage in messages.OrderBy(x => x.ServerDateUtc))
                {
                    var messageViewModel = CreateMessageViewModel(chatMessage, chatInfo);
                    if (messageViewModel != null)
                    {
                        list.Add(messageViewModel);
                    }
                }
            }

            var view = new MessagesView();
            _viewModel = new MessagesViewModel
            {
                MessagesCollection = new ObservableCollection<MessageViewModel>(list.OrderByDescending(x => x.MessageTime))
            };

            view.DataContext = _viewModel;
            _repository.SubscribeChats().Subscribe(this);
            _repository.SubscribeChatMessages().Subscribe(this);

            return view;
        }

        private MessageViewModel CreateMessageViewModel(IChatMessage chatMessage, IChatInfo chatInfo)
        {
            if (chatMessage.Type != MessageType.TextMessage)
                return null;

            var edit = chatMessage.RelatedMessages.Where(x => x.Type == MessageType.EditTextMessage)
                .OrderByDescending(x => x.ServerDateUtc).FirstOrDefault();

            var text = edit != null ? edit.GetData<ITextMessageData>().Text : chatMessage.GetData<ITextMessageData>().Text;
            var chatName = chatInfo.Chat.Type == ChatKind.Personal ?
                    _objRepository.GetPerson(int.Parse(chatInfo.Chat.Name))?.DisplayName
                    : chatInfo.Chat.Name;

            return new MessageViewModel()
            {
                MessageId = chatMessage.Id,
                ChatName = chatName,
                Text = text,
                Author = _objRepository.GetPerson(chatMessage.CreatorId)?.DisplayName,
                MessageTime = ((DateTime)(edit?.ServerDateUtc ?? chatMessage.ServerDateUtc ?? DateTime.UtcNow)).ToLocalTime()
            };
        }

        public async void OnNext(IChatMessage value)
        {
            if (_viewModel?.MessagesCollection == null)
                return;

            // Пропускаем сообщения, которые не являются текстовыми
            if (value.Type != MessageType.TextMessage && value.Type != MessageType.EditTextMessage)
                return;

            // Находим информацию о чате
            var chatInfo = _loadedChats.FirstOrDefault(c => c.Chat.Id == value.ChatId);
            if (chatInfo == null)
                return;

            // Если это редактирование сообщения
            if (value.Type == MessageType.EditTextMessage)
            {
                var originalMessageId = value.RelatedMessageId;
                if (originalMessageId == null)
                    return;

                var existingMessage = _viewModel.MessagesCollection.FirstOrDefault(m => m.MessageId == originalMessageId);
                if (existingMessage != null)
                {
                    Application.Current.Dispatcher.Invoke(() =>
                    {
                        // Обновляем текст и время сообщения
                        existingMessage.Text = value.GetData<ITextMessageData>().Text;
                        existingMessage.MessageTime = ((DateTime)(value.ServerDateUtc ?? value.ClientDateUtc)).ToLocalTime();

                        // Пересортировываем коллекцию после обновления времени
                        var allMessages = _viewModel.MessagesCollection.ToList();
                        _viewModel.MessagesCollection.Clear();
                        foreach (var msg in allMessages.OrderByDescending(x => x.MessageTime))
                        {
                            _viewModel.MessagesCollection.Add(msg);
                        }
                    });
                }
                return;
            }

            // Если это новое текстовое сообщение
            var existingNewMessage = _viewModel.MessagesCollection.FirstOrDefault(m => m.MessageId == value.Id);
            if (existingNewMessage != null)
                return; // Сообщение уже существует

            // Создаем новое сообщение
            var messageViewModel = CreateMessageViewModel(value, chatInfo);
            if (messageViewModel != null)
            {
                Application.Current.Dispatcher.Invoke(() =>
                {
                    // Находим правильную позицию для вставки (по времени)
                    var insertIndex = 0;
                    for (int i = 0; i < _viewModel.MessagesCollection.Count; i++)
                    {
                        if (_viewModel.MessagesCollection[i].MessageTime < messageViewModel.MessageTime)
                        {
                            insertIndex = i;
                            break;
                        }
                        insertIndex = i + 1;
                    }
                    
                    _viewModel.MessagesCollection.Insert(insertIndex, messageViewModel);
                });
            }
        }

        public void OnNext(IChatInfo value)
        {
            var chatInfo = _loadedChats.FirstOrDefault(c => c.Chat.Id == value.Chat.Id);
            if (chatInfo == null)
            {
                chatInfo = value;
                _loadedChats.Add(chatInfo);
            }
        }

        public void OnError(Exception error)
        {
        }

        public void OnCompleted()
        {
        }
    }
}
