/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System.ComponentModel.Composition;
using System.IO;
using System.Linq;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Input;
using Ascon.Pilot.SDK.HotKey;
using Ascon.Pilot.SDK.Menu;

namespace Ascon.Pilot.SDK.ContextMenuSample
{
    [Export(typeof(IMenu<ObjectsViewContext>)), Export(typeof(IHotKey<ObjectsViewContext>))]
    public class ObjectListContextMenuPlugin : IMenu<ObjectsViewContext>, IHotKey<ObjectsViewContext>
    {
        private const string MENU_ITEM_NAME = "PluginMenuItem";
        private const string DOCUMENT_MENU_ITEM_NAME = "DocumentPluginMenuItem";
        private const string CONTEXT_MENU_ITEM_NAME = "ContextPluginMenuItem";
        private const string MANY_SELECTED_MENU_ITEM_NAME = "ManySelectedMenuItemName";
        private const string SAVE_MENU_ITEM_NAME = "SaveExampleMenuItemName";

        private readonly IFileSaver _fileSaver;

        [ImportingConstructor]
        public ObjectListContextMenuPlugin(IFileSaver fileSaver)
        {
            _fileSaver = fileSaver;
        }

        public void Build(IMenuBuilder builder, ObjectsViewContext context)
        {
            var objects = context.SelectedObjects.ToList();
            var icon = IconLoader.GetIcon("menu_icon.svg");

            var itemNames = builder.ItemNames.ToList();
            const string indexItemName = "miShowSharingSettings";
            var insertIndex = itemNames.IndexOf(indexItemName);

            builder.AddSeparator(++insertIndex);
            // Show menu for the empty area within a folder
            if (context.IsContext)
            {
                builder.AddItem(CONTEXT_MENU_ITEM_NAME, ++insertIndex)
                       .WithHeader("Menu for empty area")
                       .WithIcon(icon);

                builder.AddItem("name", ++insertIndex)
                       .WithHeader("Disabled menu for empty area")
                       .WithIcon(icon)
                       .WithIsEnabled(false);

                builder.AddSeparator(++insertIndex);
                return;
            }

            // Show menu when a document is selected
            if (objects.Count == 1)
            {
                var selectedObject = objects.Single();
                if (selectedObject.Type.HasFiles)
                {
                    builder.AddItem(DOCUMENT_MENU_ITEM_NAME, ++insertIndex)
                           .WithHeader("Menu for document")
                           .WithIcon(icon);
                    builder.AddItem(SAVE_MENU_ITEM_NAME, insertIndex).WithHeader("Save file");
                }
                else
                {
                    builder.AddItem(MENU_ITEM_NAME, ++insertIndex)
                           .WithHeader("Menu for object")
                           .WithIcon(icon);
                }
            }

            // Show menu for many selected objects
            if (objects.Count > 1)
            {
                builder.AddItem(MANY_SELECTED_MENU_ITEM_NAME, ++insertIndex)
                       .WithHeader("Menu for many objects")
                       .WithIcon(icon);

                builder.AddItem("name", ++insertIndex)
                       .WithHeader("Disabled menu for many objects")
                       .WithIcon(icon)
                       .WithIsEnabled(false);
            }
            else
            {
                builder.AddItem("name", ++insertIndex)
                       .WithHeader("Disabled menu for object")
                       .WithIcon(icon)
                       .WithIsEnabled(false);

            }

            builder.AddSeparator(++insertIndex);
        }

        public void OnMenuItemClick(string name, ObjectsViewContext context)
        {
            if (MENU_ITEM_NAME == name)
                MessageBox.Show($"{MENU_ITEM_NAME} was clicked");

            if (DOCUMENT_MENU_ITEM_NAME == name)
                MessageBox.Show($"{DOCUMENT_MENU_ITEM_NAME} was clicked");

            if (CONTEXT_MENU_ITEM_NAME == name)
                MessageBox.Show($"{CONTEXT_MENU_ITEM_NAME} was clicked");

            if (MANY_SELECTED_MENU_ITEM_NAME == name)
                MessageBox.Show($"{MANY_SELECTED_MENU_ITEM_NAME} was clicked");

            if (name == SAVE_MENU_ITEM_NAME)
            {
                _ = SaveFileAsync(context);
            }
        }

        private async Task SaveFileAsync(ObjectsViewContext context)
        {
            var selectedObject = context.SelectedObjects.First();
            if (selectedObject == null)
                return;
            var outputFilePath = Path.GetTempFileName() + ".xps";
            var result = await _fileSaver.SaveFileAsync(selectedObject, outputFilePath);
            foreach (var res in result)
            {
                MessageBox.Show($"Saved file {res}");
            }
        }

        #region IHotKey implementation
        private const string DELETE_COMMAND_ID = "sampleExtensionDeleteCommand";
        private const string SHOW_INFO_COMMAND_ID = "sampleExtensionShowInfoCommand";

        public void AssignHotKeys(IHotKeyCollection hotKeyCollection)
        {
            hotKeyCollection.Add(Key.Delete, ModifierKeys.None, DELETE_COMMAND_ID);//overrides default hot key
            hotKeyCollection.Add(Key.I, ModifierKeys.Control, SHOW_INFO_COMMAND_ID);//adds new hot key
            hotKeyCollection.Add(Key.S, ModifierKeys.Control, SAVE_MENU_ITEM_NAME);//adds new hot key
        }

        public void OnHotKeyPressed(string commandId, ObjectsViewContext context)
        {
            switch (commandId)
            {
                case DELETE_COMMAND_ID:
                {
                    MessageBox.Show("Default delete object routine override logic here.", "Hot key extension sample.");
                    break;
                }
                case SHOW_INFO_COMMAND_ID:
                {
                    var selectedObject = context.SelectedObjects.FirstOrDefault();
                    var messasge = selectedObject != null
                        ? $"{selectedObject.DisplayName} - {selectedObject.Creator.DisplayName}"
                        : "<nothing is selected>";
                    MessageBox.Show(messasge, "Hot key extension sample.");
                    break;
                }
                case SAVE_MENU_ITEM_NAME:
                {
                    _ = SaveFileAsync(context);
                    break;
                }
            }
        }
        #endregion
    }
}
