/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Globalization;
using System.Linq;
using System.Runtime.InteropServices;
using System.Windows.Input;
using Ascon.Pilot.SDK.Controls.Commands;
using Ascon.Pilot.Theme.Tools;
using System.Threading.Tasks;
using Ascon.Pilot.SDK.Controls.ObjectCardView.ElementBook;
using System.Collections.Generic;

namespace Ascon.Pilot.SDK.Controls.ObjectCardView
{
    public class CardControlViewModel : PropertyChangedBase, ISingleValidationErrorHandler
    {
        protected readonly IType _type;
        protected readonly IPilotDialogService _dialogService;
        protected readonly IObjectsRepository _repository;
        protected readonly IAttributeFormatParser _attributeFormatParser;
        private readonly IObjectLoader _loader;
        private IElementBookConfiguration _elementBookConfiguration;
        private IElementBookUtils _elementBookUtils;
        private object _value;
        private bool _hasValidationError;
        private bool _isReadOnly;
        private readonly DValue _originalValue;
        private string _format;
        private string _culture;
        private bool _isElementBookEnabled;
        private string _elementBookTitle;

        public CardControlViewModel(IAttribute attribute, IType type, object initValue, bool isReadOnly,
            IPilotDialogService dialogService, IObjectsRepository repository, IObjectCardAutoComplete autoComplete,
            IAttributeFormatParser attributeFormatParser, ITransitionManager transitionManager, bool editMode)
        {

            _dialogService = dialogService ?? throw new ArgumentNullException(nameof(dialogService));
            _repository = repository ?? throw new ArgumentNullException(nameof(repository));
            _attributeFormatParser = attributeFormatParser ?? throw new ArgumentNullException(nameof(attributeFormatParser));
            TransitionManager = transitionManager ?? throw new ArgumentNullException(nameof(transitionManager));

            AutoComplete = autoComplete ?? throw new ArgumentNullException(nameof(autoComplete));
            Attribute = attribute ?? throw new ArgumentNullException(nameof(attribute));
            
            _type = type;
            _repository = repository;
            _attributeFormatParser = attributeFormatParser;
            _loader = new ObjectLoader(repository);
            EditMode = editMode;

            _originalValue = DValue.GetDValue(initValue);
            _value = initValue;
            IsReadOnly = isReadOnly;


            if (Attribute.Type == AttributeType.ElementBook)
                InitializeElementBook(initValue, repository);

            Format = _attributeFormatParser.GetAttributeFormat(Attribute.Configuration2());
            if (_originalValue != null && (_originalValue.DateValue != null && Format == null))
                Format = CultureInfo.CurrentCulture.DateTimeFormat.ShortDatePattern;

            Culture = _attributeFormatParser.GetAttributeFormatCulture(Attribute.Configuration2());
        }

        private void InitializeElementBook(object initValue, IObjectsRepository repository)
        {
            _attributeFormatParser.TryParseElementBookConfiguration(Attribute.Configuration2(), out _elementBookConfiguration);
            _elementBookUtils = new ElementBookUtils(repository);
            IsElementBookEnabled = _elementBookConfiguration.Description != null;
            _ = ShowElementBookTitleAsync(initValue);
        }

        private static bool IsReadOnlyReferenceBookAttribute(IAttribute attribute, IAttributeFormatParser attributeFormatParser)
        {
            if (attribute.Configuration2() == null)
                return false;

            var success = attributeFormatParser.TryParseReferenceBookConfiguration(attribute.Configuration2(), out var attributeConfiguration);
            return success && !attributeConfiguration.IsEditable;
        }

        public bool IsNotEditableReferenceBookAttribute => IsReadOnlyReferenceBookAttribute(Attribute, _attributeFormatParser);

        public IObjectsRepository Repository => _repository;

        public IObjectCardAutoComplete AutoComplete { get; }

        public ITransitionManager TransitionManager { get; }

        public IType Type => _type;

        public bool IsDataChanged
        {
            get
            {
                var dValue = DValue.GetDValue(Value);
                return !_originalValue.Equals(dValue);
            }
        }

        public IAttribute Attribute { get; }

        public IAttributeFormatParser AttributeFormatParser => _attributeFormatParser;

        public bool IsReadOnly
        {
            get => _isReadOnly;
            set
            {
                _isReadOnly = value;
                NotifyOfPropertyChange(nameof(IsReadOnly));
            }
        }

        public object Value
        {
            get => _value;
            set
            {
                _value = value;
                if (Attribute.Type == AttributeType.ElementBook)
                    _ = ShowElementBookTitleAsync(value);
                NotifyOfPropertyChange(nameof(Value));
            }
        }

        public string Format
        {
            get => _format;
            set
            {
                _format = value;
                NotifyOfPropertyChange(nameof(Format));
            }
        }

        public string Culture
        {
            get => _culture;
            set
            {
                _culture = value;
                NotifyOfPropertyChange(nameof(Culture));

            }
        }

        public bool HasValidationError
        {
            get => _hasValidationError;
            set
            {
                _hasValidationError = value;
                NotifyOfPropertyChange(nameof(HasValidationError));
            }
        }

        public string ElementBookTitle
        {
            get => _elementBookTitle;
            set
            {
                _elementBookTitle = value;
                NotifyOfPropertyChange(nameof(ElementBookTitle));
            }
        }

        public bool IsElementBookEnabled
        {
            get => _isElementBookEnabled;
            set
            {
                _isElementBookEnabled = value;
                NotifyOfPropertyChange(nameof(IsElementBookEnabled));
            }
        }

        public string Error { get; set; }

        public ICommand ShowReferenceBookViewCommand => new DelegateCommand(ShowReferenceBookView);

        public ICommand ShowElementBookViewCommand => new DelegateCommand(ShowElementBookView);

        public bool EditMode { get; }
        public IElementBookConfiguration ElementBookConfiguration { get => _elementBookConfiguration; }

        [DllImport("user32.dll")]
        public static extern IntPtr GetActiveWindow();

        private void ShowElementBookView()
        {
            var checkedNodes = (Value as string[])?.Select(n => Guid.Parse(n)).ToList() ?? new List<Guid>();

            var options = _dialogService.NewOptions();
            options.WithAllowChecking(true);
            options.WithParentWindow(GetActiveWindow());
            options.WithAllowMultiSelect(_elementBookConfiguration.Description.AllowMultiSelect);
            options.WithCheckedNodes(checkedNodes.Cast<object>());
            
            var selectedObjects = _dialogService.ShowElementBookDialog(Attribute.Configuration2(), _type.Name + "-" + Attribute.Name, options);

            if (selectedObjects != null)
                Value = selectedObjects.Select(n => n.Id.ToString()).ToArray();
        }

        private void ShowReferenceBookView()
        {
            var success = _attributeFormatParser.TryParseReferenceBookConfiguration(Attribute.Configuration2(), out var configuration);
            if (!success)
                return;

            if (configuration.Kind == RefBookKind.OrgUnit)
            {
                var dialogOptions = _dialogService.NewOptions()
                    .WithAllowChecking(true)
                    .WithAllowMultiSelect(true)
                    .WithOkButtonCaption("Ok")
                    .WithParentWindow(GetActiveWindow());

                var selectedPositions = _dialogService.ShowPositionSelectorDialog(dialogOptions);

                var positionsTitles = selectedPositions.Select(x => GetEffectiveAttributeTitle(x.Id));
                Value = string.Join("; ", positionsTitles);
                return;
            }

            var options = _dialogService.NewOptions();
            options.WithAllowChecking(true);
            options.WithParentWindow(GetActiveWindow());
            options.WithAllowMultiSelect(configuration.AllowMultiSelect);

            var checkedNodes = _dialogService.ShowReferenceBookDialog(Attribute.Configuration2(), _type.Name + "-" + Attribute.Name, options).ToList();
            if (!checkedNodes.Any())
                return;

            var titles = checkedNodes.Select(n => GetEffectiveAttributeTitle(n, configuration.StringFormat));
            Value = string.Join("; ", titles);

            if (checkedNodes.Count() == 1)
                foreach (var attribute in checkedNodes.First().Attributes)
                    AutoComplete.Fill(attribute.Key, attribute.Value);
        }

        private string GetEffectiveAttributeTitle(IDataObject obj, string stringFormat)
        {
            var formattedTitle = string.IsNullOrEmpty(stringFormat)
                ? obj.DisplayName
                : _attributeFormatParser.AttributesFormat(stringFormat, obj.Attributes.ToDictionary(x => x.Key, x => x.Value));
            return formattedTitle;
        }

        private string GetEffectiveAttributeTitle(int orgUnitId)
        {
            var people = _repository.GetPeople().ToList();
            var person = people.Where(p => p.Positions.Select(m => m.Position).Contains(orgUnitId))
                .OrderBy(o => o.Positions.First(x => x.Position == orgUnitId).Order)
                .FirstOrDefault();

            return person == null ? string.Empty : person.DisplayName;
        }

        private async Task ShowElementBookTitleAsync(object attrValue)
        {
            if (!IsElementBookEnabled)
                return;
            var ids = attrValue as string[];
            if (ids == null)
                return;

            if (!ids.Any())
            {
                ElementBookTitle = string.Empty;
                return;
            }

            var objs = await _loader.Load(ids.Select(id => Guid.Parse(id)));
            var loaded = objs.Where(o => _elementBookUtils.CanCreateElementBook(o, _elementBookConfiguration.Description));
            var titles = loaded.Select(n => _elementBookUtils.GetEffectiveAttributeTitle(n, _elementBookConfiguration.Description.StringFormat));
            ElementBookTitle = string.Join("; ", titles);
        }
    }
}
