/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.ObjectModel;
using System.ComponentModel.Composition;
using System.Windows;
using Ascon.Pilot.SDK.Controls;

namespace Ascon.Pilot.SDK.FileAttachmentTracking
{
    /// <summary>
    /// Data model for displaying information about an attached file
    /// </summary>
    public class FileAttachmentInfo : PropertyChangedBase
    {
        public string FileName { get; set; }
        public string FilePath { get; set; }
        public Guid ObjectId { get; set; }
        public string ObjectType { get; set; }
        public long FileSize { get; set; }
        public DateTime AttachedTime { get; set; }
        public string Extension { get; set; }
        public DateTime? LastModified { get; set; }
    }

    /// <summary>
    /// Main module for tracking file attachments with UI
    /// </summary>
    [Export(typeof(INewTabPage))]
    [ExportMetadata("Name", "File Attachment Tracking")]
    public class FileAttachmentTrackingModule : PropertyChangedBase, INewTabPage
    {
        private const string OPEN_FILE_TRACKING_COMMAND = "OpenFileTrackingCommand";
        private readonly ITabServiceProvider _tabServiceProvider;

        private static ObservableCollection<FileAttachmentInfo> _attachedFiles = new ObservableCollection<FileAttachmentInfo>();

        /// <summary>
        /// Collection of attached files for XAML binding
        /// </summary>
        public ObservableCollection<FileAttachmentInfo> AttachedFiles => _attachedFiles;

        /// <summary>
        /// Static method for adding file information from the handler
        /// </summary>
        public static void AddFileInfo(FileAttachmentInfo fileInfo)

        {
            // Ensure adding in the UI thread
            if (Application.Current != null)
            {
                Application.Current.Dispatcher.Invoke(() =>
                {
                    _attachedFiles.Insert(0, fileInfo); // Insert at the beginning to show recent files on top
                });
            }
            else
            {
                _attachedFiles.Insert(0, fileInfo);
            }
        }

        [ImportingConstructor]
        public FileAttachmentTrackingModule(ITabServiceProvider tabServiceProvider)
        {
            _tabServiceProvider = tabServiceProvider;
        }

        #region INewTabPage Implementation

        public void BuildNewTabPage(INewTabPageHost host)
        {
            host.AddButton("File Attachment Tracking", OPEN_FILE_TRACKING_COMMAND, 
                "Tracking file additions to object cards", null);
        }

        public void OnButtonClick(string name)
        {
            if (name != OPEN_FILE_TRACKING_COMMAND)
                return;

            var title = _tabServiceProvider.GetActiveTabPageTitle();
            var view = new FileTrackingView();
            view.DataContext = this;
            _tabServiceProvider.UpdateTabPageContent(title, "File Attachment Tracking", view);
        }

        #endregion
    }
} 