/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.IO;
using System.Linq;
using System.Windows.Input;
using System.Windows.Threading;
using Ascon.Pilot.SDK.Controls.Commands;
using Ascon.Pilot.Theme.Tools;
using Microsoft.Win32;
using Ascon.Pilot.SDK.Controls;

namespace Ascon.Pilot.SDK.FileSnapshotSample
{
    abstract class HistoryViewModel : PropertyChangedBase, IObserver<IDataObject>
    {
        protected readonly Guid _dataObjectId;
        protected readonly IObjectsRepository _repository;
        protected readonly IObjectModifier _modifier;
        private readonly Dispatcher _dispatcher;

        private readonly DelegateCommand _createVersionCommand;
        private readonly DelegateCommand<HistoryItem> _makeActualCommand;
        private HistoryItem _selectedItem;

        protected HistoryViewModel(Guid dataObjectId, IObjectsRepository repository, IObjectModifier modifier, Dispatcher dispatcher)
        {
            _dataObjectId = dataObjectId;
            _repository = repository;
            _modifier = modifier;
            _dispatcher = dispatcher;

            _createVersionCommand = new DelegateCommand(CreateVersion);
            _makeActualCommand = new DelegateCommand<HistoryItem>(MakeActual, CanMakeActual);

            repository.SubscribeObjects(new[] { dataObjectId }).Subscribe(this);
        }

        public ObservableCollection<HistoryItem> HistoryItems { get; } = new ObservableCollection<HistoryItem>();

        public HistoryItem SelectedItem
        {
            get => _selectedItem;
            set
            {
                _selectedItem = value;
                NotifyOfPropertyChange(nameof(SelectedItem));
                RaiseCommandCanExecute();
            }
        }

        public ICommand CreateVersionCommand => _createVersionCommand;

        public ICommand MakeActualCommand => _makeActualCommand;

        public abstract string Title { get; }

        private async void MakeActual(HistoryItem item)
        {
            var loader = new ObjectLoader(_repository);
            var obj = await loader.Load(_dataObjectId);

            if (ShowReasonDialog(obj, out var reason) == false)
                return;

            _modifier.Edit(obj).MakeSnapshotActual(reason, item.Snapshot);
            _modifier.Apply();
            
        }

        private bool CanMakeActual(HistoryItem item)
        {
            if (item == null || ReferenceEquals(item, HistoryItems.First()))
                return false;

            return true;
        }

        protected virtual void RaiseCommandCanExecute()
        {
            _dispatcher.BeginInvoke(new Action(() =>
            {
                if (_createVersionCommand != null)
                    _createVersionCommand.RaiseCanExecuteChanged();
                if (_makeActualCommand != null)
                    _makeActualCommand.RaiseCanExecuteChanged();
            }));
        }

        protected virtual void CreateVersion()
        {
        }

        public void OnNext(IDataObject value)
        {
            UpdateHistory(value);
        }

        public void OnError(Exception error)
        {

        }

        public void OnCompleted()
        {
        }

        private void UpdateHistory(IDataObject dataObject)
        {
            _dispatcher.BeginInvoke(new Action(() =>
            {
                HistoryItems.Clear();

                //add actual snapshot
                HistoryItems.Add(new HistoryItem(dataObject.ActualFileSnapshot, _repository));

                //add history
                foreach (var snapshot in dataObject.PreviousFileSnapshots)
                {
                    HistoryItems.Add(new HistoryItem(snapshot, _repository));
                }
            }));
        }

        protected static bool? ShowReasonDialog(IDataObject o, out string reason)
        {
            var model = new ReasonViewModel(o.DisplayName);
            var view = new ReasonView
            {
                DataContext = model
            };

            try
            {
                return view.ShowDialog();
            }
            finally
            {
                reason = model.Reason;
            }
        }

        protected bool? ShowOpenFileDialog(string title, string filter, bool mutiselect, string initialDirectory, out List<string> filePaths)
        {
            var dialog = new OpenFileDialog
            {
                Multiselect = mutiselect,
                Title = title
            };

            if (filter != null)
                dialog.Filter = filter;

            if (initialDirectory != null)
            {
                var fullPath = Path.GetFullPath(initialDirectory);
                dialog.InitialDirectory = fullPath;
            }

            try
            {
                return dialog.ShowDialog(LayoutHelper.GetOwnerWindow());
            }
            finally
            {
                filePaths = dialog.FileNames.ToList();
            }
        }
    }
}
