/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System.ComponentModel.Composition;
using System.Diagnostics;
using System.IO;
using System.Linq;
using Ascon.Pilot.SDK.Menu;

namespace Ascon.Pilot.SDK.FilesSample
{
    [Export(typeof(IMenu<ObjectsViewContext>))]
    public class FilesSamplePlugin : IMenu<ObjectsViewContext>
    {
        private readonly IFileProvider _fileProvider;
        private IDataObject _selectedObject;
        private const string MENU_ITEM_NAME = "OpenFileInExplorerMenuItemName";

        [ImportingConstructor]
        public FilesSamplePlugin(IFileProvider fileProvider, IObjectModifier modifier)
        {
            _fileProvider = fileProvider;
        }

        public void Build(IMenuBuilder builder, ObjectsViewContext context)
        {
            var itemNames = builder.ItemNames.ToList();
            const string indexItemName = "miSend";
            var insertIndex = itemNames.IndexOf(indexItemName) + 1;

            // Show menu when a document is selected
            var objects = context.SelectedObjects.ToList();
            if (objects.Count() != 1)
                return;

            _selectedObject = objects.Single();
            if (_selectedObject.Type.HasFiles && _selectedObject.Files.Count > 0)
                builder.AddItem(MENU_ITEM_NAME, insertIndex).WithHeader("Open file in Internet Explorer");
        }

        public void OnMenuItemClick(string name, ObjectsViewContext context)
        {
            if (name != MENU_ITEM_NAME) return;

            var file = _selectedObject.Files.FirstOrDefault(f => IsCorrectFileExtension(f.Name));
            if (file == null)
                return;

            if (!_fileProvider.Exists(file.Id))
                return;

            var fileName = SaveFile(file);
            StartExplorer(fileName);
        }

        private string SaveFile(IFile file)
        {
            var inputStream = _fileProvider.OpenRead(file);
            var fileName = Path.Combine(Path.GetTempPath(), file.Name);
            using (var fileStream = File.Create(fileName))
            {
                inputStream.Seek(0, SeekOrigin.Begin);
                inputStream.CopyTo(fileStream);
            }
            return fileName;
        }

        private bool IsCorrectFileExtension(string name)
        {
            var ext = Path.GetExtension(name);
            if (ext == ".xps" || ext == ".pdf")
                return true;

            return false;
        }

        private void StartExplorer(string fileName)
        {
            var p = new Process
            {
                StartInfo =
                {
                    FileName = "iexplore.exe", 
                    Arguments = fileName
                }
            };
            p.Start();
        }
    }
}
