/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.ComponentModel.Composition;
using System.Windows;

namespace Ascon.Pilot.SDK.LicenseSample
{
    [Export(typeof(IDataPlugin))]
    public class LicenseSampleMain : IDataPlugin, IObserver<bool>, IHandle<UnloadedEventArgs>, IHandle<OnlineEventArgs>
    {
        private readonly ILicenseManager _licenseManager;
        private readonly IEventAggregator _eventAggregator;
        private IDisposable _licenseObserver;
        private bool _isLicenseCaptured;
        private const int ProductId = 501;

        [ImportingConstructor]
        public LicenseSampleMain(ILicenseManager licenseManager, IEventAggregator eventAggregator)
        {
            _licenseManager = licenseManager;
            _licenseObserver = _licenseManager.Consume(ProductId).Subscribe(this);
            _eventAggregator = eventAggregator;
            _eventAggregator.Subscribe(this);
        }

        /// <summary>
        /// License  type handler
        /// </summary>
        /// <param name="licenseResult">true - valid license file exists, false - license files doesn't exist(free connections mode enabled)</param>
        public void OnNext(bool licenseResult)
        {
            if (licenseResult)
            {
                _isLicenseCaptured = true;
                MessageBox.Show("License captured");
            }
            else
                MessageBox.Show("Free connections mode");
        }

        public async void OnError(Exception error)
        {
            var licenseInfoLoader = new LicenseInfoLoader(_licenseManager);
            var licenseInfo = await licenseInfoLoader.LoadLicenseInfoAsync(ProductId);

            if (licenseInfo == null)
            {
                MessageBox.Show("Free connections mode");
                return;
            }

            if (licenseInfo.IsCheated)
            {
                MessageBox.Show("Cheated license!");
                return;
            }

            if (licenseInfo.IsExpired)
            {
                MessageBox.Show("License is expired");
                return;
            }

            if (licenseInfo.MaxLicensesCount == 0)
            {
                MessageBox.Show("Product license cannot be found");
                return;
            }

            MessageBox.Show("Overflow license count");
        }

        public void OnCompleted()
        {
            _licenseObserver?.Dispose();
            MessageBox.Show("Action completed");
        }

        public void Handle(UnloadedEventArgs message)
        {
            _licenseManager.Release(ProductId);
            _eventAggregator.Unsubscribe(this);
        }

        public void Handle(OnlineEventArgs message)
        {
            if (!_isLicenseCaptured)
                _licenseObserver = _licenseManager.Consume(ProductId).Subscribe(this);
        }
    }
}
