/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices;
using System.Threading.Tasks;
using System.Windows.Input;
using System.Windows.Markup;
using System.Xml.Linq;
using Ascon.Pilot.SDK.Controls;
using Ascon.Pilot.SDK.Controls.Commands;
using Ascon.Pilot.SDK.Data;
using static System.Net.Mime.MediaTypeNames;

namespace Ascon.Pilot.SDK.MessagesSample
{
    internal class MessagesViewModel : PropertyChangedBase
    {
        private readonly IMessagesRepository _repository;
        private readonly Guid _chatId;
        private  bool _chatExist;
        private readonly IMessageBuilder _messageBuilder;
        private readonly IObjectsRepository _objRepository;
        private readonly IPilotDialogService _pilotDialogService;
        private ICommand _sendCommand;
        private ICommand _addMemberCommand;
        private string _members;
        private List<int> _membersList = new List<int>();
        private readonly IDataObject _obj;

        public MessagesViewModel(IMessagesRepository repository, IDataObject obj, bool chatExist,
            IMessageBuilder messageBuilder, IObjectsRepository objectsRepository,
            IPilotDialogService pilotDialogService)
        {
            _repository = repository;
            _chatId = obj.Id;
            _obj = obj;
            _chatExist = chatExist;
            _messageBuilder = messageBuilder;
            _objRepository = objectsRepository;
            _pilotDialogService = pilotDialogService;
            _sendCommand = new DelegateCommand(SendMessage);
            _addMemberCommand = new DelegateCommand(AddMember);

            LoadMembers();
        }

        private async void AddMember()
        {
            var dialogOptions =
                _pilotDialogService.NewOptions()
                    .WithParentWindow(GetActiveWindow())
                    .WithCaption("Select members")
                    .WithOkButtonCaption("Select")
                    .WithAllowChecking(true)
                    .WithAllowMultiSelect(true);

            var members = _pilotDialogService.ShowOrganisationUnitSelectorDialog(dialogOptions).Where(x => x.Children.Count == 0 && !_membersList.Contains(x.Id)).ToList();
            if (!members.Any())
                return;

            if (!_chatExist)
                CreateChat();

            _messageBuilder.CreateMessage(MessageType.ChatMembers, _chatId);
            foreach (var member in members)
            {
                _messageBuilder.AddChatMemberData(member.Person(), true, false);
            }
            await _repository.SendMessageAsync(_messageBuilder.Message);
            LoadMembers();
        }

        [DllImport("user32.dll")]
        public static extern IntPtr GetActiveWindow();

        private async void LoadMembers()
        {
            var people = _objRepository.GetPeople().ToDictionary(x => x.Id, person => person);

            var members = await _repository.LoadChatMembersAsync(_chatId, DateTime.MinValue);
            Members = String.Join(", ", members.Select(x => people.TryGetValue(x.PersonId, out var person) ? person.DisplayName : null));
            _membersList = new List<int>();
            _membersList.AddRange(members.Select(x=>x.PersonId));
        }

        private async void SendMessage()
        {
            if (!_chatExist)
                CreateChat();

            var text = NewText;
            bool isReadOnly = text.StartsWith("!");
            var message = _messageBuilder.CreateMessage(MessageType.TextMessage, _chatId)
                .AddText(NewText, isReadOnly)
                .Message;
            
            await _repository.SendMessageAsync(message);

            MessagesCollection.Add(new MessageViewModel()
            {
                Text = NewText,
                Name = _objRepository.GetPerson(message.CreatorId)?.DisplayName
            });
            NewText = "";
            NotifyOfPropertyChange(nameof(NewText));
        }

        private async void CreateChat()
        {
            var message = _messageBuilder.CreateMessage(MessageType.ChatCreation, _chatId).
                AddChatCreationData(_obj.DisplayName, "", ChatKind.ObjectRelated).Message;
            
          await  _repository.SendMessageAsync(message);

            SendRelation();

            _chatExist = true;
        }

        private async void SendRelation()
        {
            var message = _messageBuilder.CreateMessage(MessageType.ChatRelation, _chatId).
                AddChatRelation(ChatRelationType.Relation,_chatId).Message;

            await _repository.SendMessageAsync(message);
        }

        public ObservableCollection<MessageViewModel> MessagesCollection { get; set; }

        public string NewText { get; set; }

        public string Members
        {
            get => _members;
            set
            {
                _members = value; 
                NotifyOfPropertyChange(nameof(Members));
            }
        }

        public ICommand SendCommand => _sendCommand;
        public ICommand AddMemberCommand => _addMemberCommand;
    }
}
