/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.ObjectModel;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Windows;
using System.Windows.Data;
using System.Windows.Input;
using Ascon.Pilot.SDK.ObjectCard;
using Ascon.Pilot.SDK.ObjectCardControlSample.Annotations;
using Ascon.Pilot.Theme.Tools;

namespace Ascon.Pilot.SDK.ObjectCardControlSample
{
    public class ExtensionNewObjectDialogViewModel : INotifyPropertyChanged
    {
        private readonly IObjectsRepository _repository;
        private readonly IObjectCardControlProvider _cardControlProvider;
        private readonly IObjectModifier _modifier;
        private readonly Guid _parentObjectId;
        private readonly ActionCommand _createCommand;

        private ObjectCardControl _objectCardControl;
        private IType _selectedType;
        private Guid _newObjectId;
        private bool _canCreate;

        private bool? _closeResult;

        public ExtensionNewObjectDialogViewModel(
            IObjectsRepository repository,
            IObjectCardControlProvider cardControlProvider,
            IObjectModifier modifier,
            Guid parentObjectId)
        {
            _repository = repository;
            _cardControlProvider = cardControlProvider;
            _modifier = modifier;
            _parentObjectId = parentObjectId;

            foreach (var type in _repository.GetTypes().Where(x => !x.IsDeleted && x.Kind != TypeKind.System && x.DisplayAttributes.Any()))
            {
                ObjectsTypes.Add(type);
            }
            SortCollection(ObjectsTypes, "Sort", ListSortDirection.Ascending);
            SelectedType = ObjectsTypes.FirstOrDefault();

            _createCommand = new ActionCommand((p) => Create());
        }

        public ObservableCollection<IType> ObjectsTypes { get; } = new ObservableCollection<IType>();

        private void SortCollection<T>(ObservableCollection<T> collection, string sortProperty, ListSortDirection sortDirection)
        {
            ICollectionView view = CollectionViewSource.GetDefaultView(collection);
            view.SortDescriptions.Add(new SortDescription(sortProperty, sortDirection));
        }

        public IType SelectedType
        {
            get => _selectedType;
            set
            {
                _selectedType = value;
                OnPropertyChanged(nameof(SelectedType));
                OnSelectedTypeChanged();
            }
        }

        public ObjectCardControl ObjectCardControl
        {
            get => _objectCardControl;
            set
            {
                if (_objectCardControl != null)
                    _objectCardControl.Interaction.OnObjectCardUpdated -= OnObjectCardUpdated;

                _objectCardControl = value;

                if (_objectCardControl != null)
                    _objectCardControl.Interaction.OnObjectCardUpdated += OnObjectCardUpdated;

                OnPropertyChanged(nameof(ObjectCardControl));
            }
        }

        public ICommand CreateCommand => _createCommand;

        public bool CanCreate
        {
            get => _canCreate;
            set
            {
                _canCreate = value;
                OnPropertyChanged(nameof(CanCreate));
            }
        }

        public bool? CloseResult
        {
            get => _closeResult;
            set
            {
                _closeResult = value;
                OnPropertyChanged(nameof(CloseResult));
            }
        }

        private void OnSelectedTypeChanged()
        {
            _newObjectId = Guid.NewGuid();

            ObjectCardControl = _cardControlProvider.GetObjectCardControl(
                newObjectId: _newObjectId,
                newObjectType: SelectedType,
                parentObjectId: _parentObjectId,
                initAttributes: null);

            OnObjectCardUpdated(null, EventArgs.Empty);
        }

        private void Create()
        {
            try
            {
                var builder = _modifier.Create(_parentObjectId, SelectedType);
                foreach (var objectCardValue in _objectCardControl.Interaction.GetValues())
                {
                    builder.SetAttributeAsObject(objectCardValue.Key, objectCardValue.Value.Value);

                    foreach (var valueLinkTargetId in objectCardValue.Value.LinkTargetIds)
                    {
                        builder.AddRelation(
                            objectCardValue.Key,
                            ObjectRelationType.Custom,
                            _newObjectId,
                            valueLinkTargetId,
                            default(DateTime));
                    }

                }

                _modifier.Apply();
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.ToString(), "Unable to create object.");
            }

            CloseResult = true;
        }

        private void OnObjectCardUpdated(object sender, EventArgs e)
        {
            CanCreate = ObjectCardControl.Interaction.GetIsValidInput();
        }

        public event PropertyChangedEventHandler PropertyChanged;

        [NotifyPropertyChangedInvocator]
        protected virtual void OnPropertyChanged(string propertyName)
        {
            PropertyChanged?.Invoke(this, new PropertyChangedEventArgs(propertyName));
        }
    }
}
