/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System.ComponentModel.Composition;
using System.Linq;
using System.Net.Mime;
using System.Windows;
using System.Windows.Media;
using Ascon.Pilot.SDK.Menu;
using Ascon.Pilot.SDK.ObjectCard;
using Ascon.Pilot.Theme.ColorScheme;

namespace Ascon.Pilot.SDK.ObjectCardControlSample
{
    [Export(typeof(IMenu<ObjectsViewContext>))]
    public class ObjectCardControlSample : IMenu<ObjectsViewContext>
    {
        private const string SHOW_EXTENSION_OBJECT_CARD = "ShowExtensionObjectCard";
        private const string SHOW_EXTENSION_NEW_OBJECT_OBJECT_CARD = "ShowExtensionNewObjectObjectCard";

        private readonly IPilotDialogService _dialogService;
        private readonly IObjectsRepository _repository;
        private readonly IObjectCardControlProvider _objectCardControlProvider;
        private readonly IObjectModifier _modifier;

        [ImportingConstructor]
        public ObjectCardControlSample(
            IPilotDialogService dialogService, 
            IObjectsRepository repository,
            IObjectCardControlProvider objectCardControlProvider,
            IObjectModifier modifier)
        {
            _dialogService = dialogService;
            _repository = repository;
            _objectCardControlProvider = objectCardControlProvider;
            _modifier = modifier;
            var accentColor = (Color)ColorConverter.ConvertFromString(dialogService.AccentColor);
            ColorScheme.Initialize(accentColor, dialogService.Theme);
        }

        public void Build(IMenuBuilder builder, ObjectsViewContext context)
        {
            var objects = context.SelectedObjects.ToList();

            if (objects.Count == 1)
            {
                builder.AddItem(SHOW_EXTENSION_OBJECT_CARD, builder.ItemNames.Count())
                    .WithHeader("Show extension object card");
            }

            if (context.IsContext)
            {
                builder.AddItem(SHOW_EXTENSION_NEW_OBJECT_OBJECT_CARD, builder.ItemNames.Count())
                    .WithHeader("Show extension new object dialog");
            }
        }

        public void OnMenuItemClick(string name, ObjectsViewContext context)
        {
            if (SHOW_EXTENSION_OBJECT_CARD == name)
            {
                var selectedObject = context.SelectedObjects.ToList().First();

                var objectCardDialog = new ExtensionObjectCardDialog
                {
                    DataContext = new ExtensionObjectCardDialogViewModel(
                        _repository,
                        _objectCardControlProvider,
                        selectedObject.Id)
                };
                objectCardDialog.Owner = Application.Current.MainWindow;
                objectCardDialog.ShowDialog();
            }

            if (SHOW_EXTENSION_NEW_OBJECT_OBJECT_CARD == name)
            {
                var parentObjectId = context.SelectedObjects.First().Id;

                var objectCardDialog = new ExtensionNewObjectDialog
                {
                    DataContext = new ExtensionNewObjectDialogViewModel(
                        repository: _repository,
                        cardControlProvider: _objectCardControlProvider,
                        modifier: _modifier,
                        parentObjectId: parentObjectId)
                };
                objectCardDialog.Owner = Application.Current.MainWindow;
                objectCardDialog.ShowDialog();
            }
        }
    }
}
