/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Globalization;
using System.Linq;
using System.Runtime.InteropServices;
using System.Threading.Tasks;
using System.Windows.Documents;
using System.Windows.Input;
using Ascon.Pilot.SDK.Controls;
using Ascon.Pilot.SDK.Controls.Commands;
using Ascon.Pilot.SDK.Controls.ObjectCardView.ElementBook;
using Ascon.Pilot.Theme.Tools;

namespace Ascon.Pilot.SDK.ObjectsSample.ObjectCardView
{
    class CardControlViewModel : PropertyChangedBase, ISingleValidationErrorHandler
    {
        private IType _type;
        private readonly IPilotDialogService _dialogService;
        private readonly IObjectsRepository _repository;
        private readonly IAttributeFormatParser _attributeFormatParser;
        private IObjectCardAutoComplete _autoComplete;
        private IElementBookConfiguration _elementBookConfiguration;
        private IElementBookUtils _elementBookUtils;
        private IObjectLoader _loader;
        private object _value;
        private bool _hasValidationError;
        private bool _isReadOnly;
        private readonly DValue _originalValue;
        private string _format;
        private string _culture; 
        private bool _isElementBookEnabled;
        private string _elementBookTitle;
        private bool _editMode;

        public CardControlViewModel(IAttribute attribute, IType type, object initValue, bool isReadOnly, IPilotDialogService dialogService, IObjectsRepository repository, IObjectCardAutoComplete autoComplete, IAttributeFormatParser attributeFormatParser, bool editMode)
        {
            if (dialogService == null)
                throw new ArgumentNullException("dialogService");
            if (repository == null) 
                throw new ArgumentNullException("repository");
            if (attributeFormatParser == null)
                throw new ArgumentNullException("attributeFormatParser");
            if (autoComplete == null)
                throw new ArgumentNullException("autoComplete");
            if (attribute == null)
                throw new ArgumentNullException("attribute");

            _type = type;
            _dialogService = dialogService;
            _repository = repository;
            _attributeFormatParser = attributeFormatParser;
            _autoComplete = autoComplete;
            _editMode = editMode;
            _loader = new ObjectLoader(repository);

            Attribute = attribute;
            _originalValue = DValue.GetDValue(initValue);
            _value = initValue;
            IsReadOnly = isReadOnly;

            if (Attribute.Type == AttributeType.ElementBook)
                InitializeElementBook(initValue, repository);

            Format = _attributeFormatParser.GetAttributeFormat(Attribute.Configuration2());
            if (_originalValue != null && (_originalValue.DateValue != null && Format == null))
                Format = CultureInfo.CurrentCulture.DateTimeFormat.ShortDatePattern;

            Culture = _attributeFormatParser.GetAttributeFormatCulture(Attribute.Configuration2());
        }

        private static bool IsReadOnlyReferenceBookAttribute(IAttribute attribute, IAttributeFormatParser attributeFormatParser)
        {
            if (attribute.Configuration2() == null)
                return false;

            var success = attributeFormatParser.TryParseReferenceBookConfiguration(attribute.Configuration2(), out var attributeConfiguration);

            return success && !attributeConfiguration.IsEditable;
        }

        public bool IsNotEditableReferenceBookAttribute
        {
            get { return IsReadOnlyReferenceBookAttribute(Attribute, _attributeFormatParser); }
        }

        public IObjectsRepository Repository
        {
            get { return _repository; }
        }

        public IObjectCardAutoComplete AutoComplete
        {
            get { return _autoComplete; }
        }

        public bool IsDataChanged
        {
            get
            {
                var dValue = DValue.GetDValue(Value);
                return !_originalValue.Equals(dValue);
            }
        }
        public string ElementBookTitle
        {
            get => _elementBookTitle;
            set
            {
                _elementBookTitle = value;
                NotifyOfPropertyChange(nameof(ElementBookTitle));
            }
        }

        public bool IsElementBookEnabled
        {
            get => _isElementBookEnabled;
            set
            {
                _isElementBookEnabled = value;
                NotifyOfPropertyChange(nameof(IsElementBookEnabled));
            }
        }

        public IType Type { get { return _type; } }

        public IAttribute Attribute { get; private set; }

        public IAttributeFormatParser AttributeFormatParser { get { return _attributeFormatParser; } }

        public bool IsReadOnly
        {
            get { return _isReadOnly; }
            set
            {
                _isReadOnly = value;
                NotifyOfPropertyChange("IsReadOnly");
            }
        }

        public object Value
        {
            get { return _value; }
            set
            {
                _value = value;
                if (Attribute.Type == AttributeType.ElementBook)
                    _ = ShowElementBookTitleAsync(value);
                NotifyOfPropertyChange("Value");
            }
        }

        public string Format
        {
            get { return _format; }
            set
            {
                _format = value;
                NotifyOfPropertyChange("Format");
            }
        }

        public string Culture
        {
            get { return _culture; }
            set
            {
                _culture = value;
                NotifyOfPropertyChange("Culture");
            }
        }

        public bool HasValidationError
        {
            get { return _hasValidationError; }
            set
            {
                _hasValidationError = value;
                NotifyOfPropertyChange("HasValidationError");
            }
        }

        public string Error { get; set; }

        public ICommand ShowReferenceBookViewCommand
        {
            get { return new DelegateCommand(ShowReferenceBookView); }
        }

        public ICommand ShowElementBookViewCommand => new DelegateCommand(ShowElementBookView);

        public bool EditMode
        {
            get { return _editMode; }
        }

        [DllImport("user32.dll")]
        public static extern IntPtr GetActiveWindow();

        private void ShowReferenceBookView()
        {
            IReferenceBookConfiguration configuration;
            var success = _attributeFormatParser.TryParseReferenceBookConfiguration(Attribute.Configuration2(), out configuration);
            if(!success)
                return;

            if (configuration.Kind == RefBookKind.OrgUnit)
            {
                var dialogOptions = _dialogService.NewOptions()
                    .WithAllowChecking(true)
                    .WithAllowMultiSelect(true)
                    .WithOkButtonCaption("Ok")
                    .WithParentWindow(GetActiveWindow());

                var selectedPositions = _dialogService.ShowPositionSelectorDialog(dialogOptions);

                var positionsTitles = selectedPositions.Select(x => GetEffectiveAttributeTitle(x.Id));
                Value = string.Join("; ", positionsTitles);
                return;
            }

            var checkedNodes = _dialogService.ShowReferenceBookDialog(Attribute.Configuration2(), _type.Name + "-" + Attribute.Name,
                _dialogService.NewOptions().WithParentWindow(GetActiveWindow()).WithAllowChecking(true)).ToList();

            if (!checkedNodes.Any())
                return;

            var titles = checkedNodes.Select(n => GetEffectiveAttributeTitle(n, configuration.StringFormat));
            Value = string.Join("; ", titles);

            if (checkedNodes.Count() == 1)
                foreach (var attribute in checkedNodes.First().Attributes)
                    AutoComplete.Fill(attribute.Key, attribute.Value);
        }

        private void InitializeElementBook(object initValue, IObjectsRepository repository)
        {
            _attributeFormatParser.TryParseElementBookConfiguration(Attribute.Configuration2(), out _elementBookConfiguration);
            _elementBookUtils = new ElementBookUtils(repository);
            IsElementBookEnabled = _elementBookConfiguration.Description != null;
            _ = ShowElementBookTitleAsync(initValue);
        }

        private string GetEffectiveAttributeTitle(IDataObject obj, string stringFormat)
        {
            var formattedTitle = string.IsNullOrEmpty(stringFormat)
                ? obj.DisplayName
                : _attributeFormatParser.AttributesFormat(stringFormat, obj.Attributes.ToDictionary(x=>x.Key, x=>x.Value));
            return formattedTitle;
        }

        private string GetEffectiveAttributeTitle(int orgUnitId)
        {
            var people = _repository.GetPeople().ToList();
            var person = people.Where(p => p.Positions.Select(m => m.Position).Contains(orgUnitId))
                .OrderBy(o => o.Positions.First(x => x.Position == orgUnitId).Order)
                .FirstOrDefault();

            return person == null ? string.Empty : person.DisplayName;
        }
        private void ShowElementBookView()
        {
            var checkedNodes = (Value as string[])?.Select(n => Guid.Parse(n)).ToList() ?? new List<Guid>();

            var options = _dialogService.NewOptions();
            options.WithAllowChecking(true);
            options.WithParentWindow(GetActiveWindow());
            options.WithAllowMultiSelect(_elementBookConfiguration.Description.AllowMultiSelect);
            options.WithCheckedNodes(checkedNodes.Cast<object>());

            var selectedObjects = _dialogService.ShowElementBookDialog(Attribute.Configuration2(), _type.Name + "-" + Attribute.Name, options);

            if (selectedObjects != null)
                Value = selectedObjects.Select(n => n.Id.ToString()).ToArray();
        }

        private async Task ShowElementBookTitleAsync(object attrValue)
        {
            if (!IsElementBookEnabled)
                return;
            var ids = attrValue as string[];
            if (ids == null)
                return;

            if (!ids.Any())
            {
                ElementBookTitle = string.Empty;
                return;
            }

            var objs = await _loader.Load(ids.Select(id => Guid.Parse(id)));
            var loaded = objs.Where(o => _elementBookUtils.CanCreateElementBook(o, _elementBookConfiguration.Description));
            var titles = loaded.Select(n => _elementBookUtils.GetEffectiveAttributeTitle(n, _elementBookConfiguration.Description.StringFormat));
            ElementBookTitle = string.Join("; ", titles);
        }
    }
}
