/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Linq;
using Ascon.Pilot.SDK.Menu;
using System.ComponentModel.Composition;
using Microsoft.Win32;

namespace Ascon.Pilot.SDK.Relations
{
    [Export(typeof(IMenu<ObjectsViewContext>))]
    public class RelationManager : IMenu<ObjectsViewContext>
    {
        private const string CUSTOM_RELATION_COMMAND = "CustomRelation";
        private const string SOURCE_FILE_RELATION_COMMAND = "SourceFileRelation";
        private const string TASK_INITIATOR_ATTACHMENT_RELATION_COMMAND = "TaskInitiatorAttachmentRelation";

        private readonly IObjectModifier _modifier;
        private readonly IPilotDialogService _dialogService;
        private readonly IObjectsRepository _repository;

        [ImportingConstructor]
        public RelationManager(IObjectModifier modifier, IPilotDialogService dialogService, IObjectsRepository repository)
        {
            _modifier = modifier;
            _dialogService = dialogService;
            _repository = repository;
        }

        public void Build(IMenuBuilder builder, ObjectsViewContext context)
        {
            if (context.SelectedObjects.Count() != 1)
                return;

            var menu = builder.AddItem("miRealtions", 1).WithHeader("Relation manager").WithSubmenu();
            menu.AddItem(CUSTOM_RELATION_COMMAND, 0).WithHeader("Add custom relation");
            var selected = context.SelectedObjects.First();
            menu.AddItem(SOURCE_FILE_RELATION_COMMAND, 1).WithHeader("Add source relation").WithIsEnabled(selected.Type.HasFiles);
            menu.AddItem(TASK_INITIATOR_ATTACHMENT_RELATION_COMMAND, 2).WithHeader("Add initiator attachment");
        }

        public void OnMenuItemClick(string name, ObjectsViewContext context)
        {
            if (name == CUSTOM_RELATION_COMMAND)
            {
                CreateCustomLink(context);
            }

            if (name == SOURCE_FILE_RELATION_COMMAND)
            {
                CreateSourceFileLink(context);
            }

            if (name == TASK_INITIATOR_ATTACHMENT_RELATION_COMMAND)
            {
                CreateTaskAttachmentLink(context);
            }
        }

        private void CreateSourceFileLink(ObjectsViewContext context)
        {
            var root = _repository.GetStoragePath();
            var openFileDialog = new OpenFileDialog
            {
                Multiselect = false,
                ShowReadOnly = true,
                InitialDirectory = root
            };

            if (openFileDialog.ShowDialog() == true)
            {
                var chosenPaths = openFileDialog.FileNames;
                var chosen = _repository.GetStorageObjects(chosenPaths).First();
                var selected = context.SelectedObjects.First();

                foreach (var relation in selected.Relations.Where(x => x.Type == ObjectRelationType.SourceFiles))
                {
                    _modifier.RemoveLink(selected, relation);
                }

                //Create relations
                var relationId = Guid.NewGuid();
                var relationName = relationId.ToString();
                var relationType = ObjectRelationType.SourceFiles;

                var selectedRealtion = new Relation
                {
                    Id = relationId,
                    Type = relationType,
                    Name = relationName,
                    TargetId = chosen.DataObject.Id
                };

                var chosenRelation = new Relation
                {
                    Id = relationId,
                    Type = relationType,
                    Name = relationName,
                    TargetId = selected.Id
                };

                _modifier.CreateLink(selectedRealtion, chosenRelation);
                _modifier.Apply();
            }
        }

        private void CreateCustomLink(ObjectsViewContext context)
        {
            var options = _dialogService.NewOptions();
            options.WithAllowMultiSelect(false);
            options.WithCaption("Select element for custom relation");
            options.WithOkButtonCaption("Select");

            var chosenObjects = _dialogService.ShowDocumentsSelectorDialog(options).ToList();
            if (chosenObjects.Count == 1)
            {
                var selected = context.SelectedObjects.First();
                var chosen = chosenObjects.First();

                var relationId = Guid.NewGuid();
                var relationName = relationId.ToString();
                var relationType = ObjectRelationType.Custom;

                var selectedRealtion = new Relation
                {
                    Id = relationId,
                    Type = relationType,
                    Name = relationName,
                    TargetId = chosen.Id
                };

                var chosenRelation = new Relation
                {
                    Id = relationId,
                    Type = relationType,
                    Name = relationName,
                    TargetId = selected.Id
                };

                _modifier.CreateLink(selectedRealtion, chosenRelation);
                _modifier.Apply();
            }
        }

        private void CreateTaskAttachmentLink(ObjectsViewContext context)
        {
            var options = _dialogService.NewOptions();
            options.WithAllowMultiSelect(false);
            options.WithCaption("Select a task");
            options.WithOkButtonCaption("Select");

            var chosenObjects = _dialogService.ShowTasksSelectorDialog2(options).ToList();
            if (chosenObjects.Count == 1)
            {
                var selected = context.SelectedObjects.First();
                var chosen = chosenObjects.First();
                if (chosen.Creator.Id == _repository.GetCurrentPerson().Id)
                {
                    var relation = new Relation
                    {
                        Id = Guid.NewGuid(),
                        TargetId = selected.Id,
                        Type = ObjectRelationType.TaskAttachments
                    };

                    var backRelation = new Relation
                    {
                        Id = relation.Id,
                        TargetId = chosen.Id,
                        Type = relation.Type,
                        Name = relation.Name
                    };

                    _modifier.CreateLink(relation, backRelation);
                    _modifier.Apply();
                }
            }
        }
    }
}
