/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.ComponentModel.Composition;
using System.Linq;
using System.Runtime.InteropServices;
using System.Windows;
using System.Windows.Interop;
using System.Windows.Media;
using Ascon.Pilot.SDK.Menu;
using Ascon.Pilot.Theme.ColorScheme;
using Ascon.Pilot.Theme.Controls;

namespace Ascon.Pilot.SDK.SearchSample.SourceFileSearch
{
    [Export(typeof(IMenu<StorageContext>))]
    class StorageContextMenu : IMenu<StorageContext>
    {
        [DllImport("user32.dll")]
        public static extern IntPtr GetForegroundWindow();

        private readonly ISearchService _searchService;
        private readonly IObjectsRepository _repository;
        private readonly ITabServiceProvider _tabServiceProvider;

        private const string PrevItemName = "mniShowProjectsExplorerCommand";
        private const string SearchFilesItemName = "mniSearchSearchFiles";

        [ImportingConstructor]
        public StorageContextMenu(ISearchService searchService, IPilotDialogService pilotDialogService, IObjectsRepository repository, ITabServiceProvider tabServiceProvider)
        {
            _searchService = searchService;
            _repository = repository;
            _tabServiceProvider = tabServiceProvider;

            InitColorScheme(pilotDialogService);
        }

        public void Build(IMenuBuilder builder, StorageContext context)
        {
            var searchContextFolder = GetSelectedFolder(context.SelectedObjects);
            if (searchContextFolder == null)
                return;

            var insertIndex = builder.ItemNames.ToList().IndexOf(PrevItemName) + 1;
            builder.AddItem(SearchFilesItemName, insertIndex).WithHeader("Поиск файлов");
        }

        public void OnMenuItemClick(string name, StorageContext context)
        {
            if (name != SearchFilesItemName)
                return;

            var searchContextFolder = GetSelectedFolder(context.SelectedObjects);
            if (searchContextFolder == null)
                return;

            var searchViewModel = new FileSearchViewModel(_repository, _searchService, searchContextFolder, _tabServiceProvider);
            var searchView = new SearchResultsView { DataContext = searchViewModel };

            var searchViewDialog = new DialogWindow
            {
                Content = searchView,
                Title = "Поиск файлов",
                Width = 500,
                Height = 600,
                WindowStartupLocation = WindowStartupLocation.CenterScreen,
                ShowInTaskbar = true
            };

            ShowAboveActiveWindow(searchViewDialog);
        }

        private IStorageDataObject GetSelectedFolder(IEnumerable<IStorageDataObject> selectedItems)
        {
            var selection = selectedItems.ToList();
            var selectedFolder = selection.Count == 1 && (selection[0].IsDirectory || selection[0].Path == _repository.GetStoragePath())
                ? selection[0]
                : null;

            return selectedFolder;
        }

        private void ShowAboveActiveWindow(DialogWindow windowToShow)
        {
            var activeWindowHandle = GetForegroundWindow();
            var wih = new WindowInteropHelper(windowToShow)
            {
                Owner = activeWindowHandle
            };

            windowToShow.Show();
            wih.Owner = IntPtr.Zero;
        }

        private void InitColorScheme(IPilotDialogService pilotDialogService)
        {
            var accentColorObj = ColorConverter.ConvertFromString(pilotDialogService.AccentColor);
            if (accentColorObj == null)
                return;

            var accentColor = (Color)accentColorObj;
            ColorScheme.Initialize(accentColor, pilotDialogService.Theme);
        }
    }
}
