/*
  Copyright © 2018 ASCON-Design Systems LLC. All rights reserved.
  This sample is licensed under the MIT License.
*/
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Ascon.Pilot.SDK.Controls;
using Ascon.Pilot.SDK.Controls.ObjectCardView;
using Ascon.Pilot.SDK.TaskSample.Services;
using Ascon.Pilot.SDK.TaskSample.TaskEditView.Attachments;
using Ascon.Pilot.SDK.TaskSample.TaskEditView.ObjectCard;

namespace Ascon.Pilot.SDK.TaskSample.TaskEditView.TaskObjectCard
{
    public class TaskCardControlViewModel : PropertyChangedBase
    {
        private protected IPilotServices _pilotServices;
        private readonly IInvalidateData _invalidateData;

        private ObjectCardViewModel _taskCardViewModel;
        private AttachmentsViewModel _attachmentsViewModel;

        public TaskCardControlViewModel(IType type, IPilotServices pilotServices, IInvalidateData invalidateData)
        {
            _pilotServices = pilotServices;
            _invalidateData = invalidateData;
            BuildCard(type);
        }

        public ObjectCardViewModel TaskCardViewModel
        {
            get => _taskCardViewModel;
            set
            {
                _taskCardViewModel = value;
                NotifyOfPropertyChange(nameof(TaskCardControl));
            }
        }

        public AttachmentsViewModel AttachmentsViewModel
        {
            get => _attachmentsViewModel;
            set
            {
                _attachmentsViewModel = value;
                NotifyOfPropertyChange(nameof(AttachmentsViewModel));
            }
        }

        protected virtual ObjectCardViewModel CreateObjectCardViewModel(IObjectsRepository repository, IPilotDialogService pilotDialogService, IAttributeFormatParser attributeFormatParser, ITransitionManager transitionManager)
        {
            return new TaskObjectCardViewModel(_pilotServices.ObjectsRepository,_pilotServices.PilotDialogService, _pilotServices.AttributeFormatParser, _pilotServices.TransitionManager);
        }

        private void BuildCard(IType type)
        {
            TaskCardViewModel = CreateObjectCardViewModel(_pilotServices.ObjectsRepository, _pilotServices.PilotDialogService, _pilotServices.AttributeFormatParser, _pilotServices.TransitionManager);
            TaskCardViewModel.CreateCard(type);
            TaskCardViewModel.IsValidInputChanged += OnIsValidInputChanged;

            AttachmentsViewModel = new AttachmentsViewModel(_pilotServices.ObjectsRepository, _pilotServices.PilotDialogService, _pilotServices.TabServiceProvider);
            AttachmentsViewModel.AttachmentsChanged += OnAttachmentsChanged;
        }

        private void OnIsValidInputChanged(object sender, EventArgs e)
        {
            Invalidate();
        }

        private void OnAttachmentsChanged(object sender, EventArgs e)
        {
            Invalidate();
            SetElementBookValueByAutoFill();
        }

        private void Invalidate()
        {
            _invalidateData.Invalidate();
        }
        private void SetElementBookValueByAutoFill()
        {
            if (!AttachmentsViewModel.Attachments.Any())
                return;

            var attachmentId = AttachmentsViewModel.Attachments.Last().Id;
            var attrsToFill = TaskCardViewModel.ViewModelCollection.Where(x => x.Attribute.Type == AttributeType.ElementBook && x.ElementBookConfiguration.Description != null && x.ElementBookConfiguration.AutoFill != null).ToList();

            foreach (var attr in attrsToFill)
                _ = SetElementBookValue(attachmentId, attr);
        }
        private async Task SetElementBookValue(Guid attachmentId, CardControlViewModel attrToFill)
        {
            if (attrToFill.IsReadOnly)
                return;

            var value = await _pilotServices.ElementBookAutoFillWorker.ComputeElementBookValueAsync(attrToFill.Attribute.Configuration2(), attachmentId);

            if (value == null) return;

            if (attrToFill.ElementBookConfiguration.Description.AllowMultiSelect)
            {
                if (attrToFill.Value is string[] previous)
                    value = previous.Concat(value).Distinct().ToArray();
            }

            attrToFill.Value = value; 
        }
    }
}
