﻿using Ascon.Pilot.SDK;
using Ascon.Pilot.SDK.Controls.Commands;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Input;

namespace BatchDigitalSigner
{
    public class ProgressDialogModel : INotifyPropertyChanged
    {
        private ProgressDialog _progressView;
        private int _signedDocsCount;
        private List<IDataObject> _selectedObjects;
        private IDigitalSigner _signer;
        private IFileProvider _fileProvider;
        private IEnumerable<int> _suitablePositions;
        private ICertificate _certificate;
        private CancellationTokenSource _cancelToken;

        public ProgressDialogModel(List<IDataObject> selectedObjects, IDigitalSigner signer, IFileProvider fileProvider, IEnumerable<int> positions, ICertificate certificate)
        {
            _selectedObjects = selectedObjects;
            _signer = signer;
            _fileProvider = fileProvider;
            _suitablePositions = positions;
            _certificate = certificate;
            _cancelToken = new CancellationTokenSource(); 
            CloseCommand = new DelegateCommand(() => _progressView.Close());
        }
        public ICommand CloseCommand { get; private set; }

        public event PropertyChangedEventHandler PropertyChanged;
        public int TotalDocsCount => _selectedObjects.Count;
        public int SignedDocsCount
        {
            get { return _signedDocsCount; }
            set
            {
                _signedDocsCount = value;
                OnPropertyChanged(nameof(SignedDocsCount));
            }
        }

        public bool IsInProgress => SignedDocsCount != TotalDocsCount;

        protected virtual void OnPropertyChanged(string propertyName)
        {
            var handler = PropertyChanged;
            handler?.Invoke(this, new PropertyChangedEventArgs(propertyName));
        }


        public void Run()
        {
            _progressView = new ProgressDialog(_cancelToken) { DataContext = this };
            Task.Run(() => 
            {
                foreach (var obj in _selectedObjects)
                {
                    if (_cancelToken.IsCancellationRequested)
                        return;

                    Sign(obj);
                    System.Windows.Application.Current.Dispatcher.Invoke(() => SignedDocsCount++);
                }
                OnPropertyChanged(nameof(IsInProgress));
            });
            _progressView.ShowDialog();

        }

        private void Sign(IDataObject obj)
        {
            var file = obj.ActualFileSnapshot.Files.FirstOrDefault(x => x.SignatureRequests.Any(sr => !sr.Signs.Any()));
            if (file == null)
                return;

            using (var stream = _fileProvider.OpenRead(file))
            {
                var signatureRequests = file.SignatureRequests.Where(x => _suitablePositions.Contains(x.PositionId) && !x.Signs.Any()).ToList();
                if (!signatureRequests.Any())
                    return;
                try
                {
                    _signer.Sign(obj.Id, file, stream, _certificate, signatureRequests);
                }
                catch (Exception ex) 
                {
                    ExceptionDialogResult? result = null;
                    System.Windows.Application.Current.Dispatcher.Invoke(() => 
                    {
                        var exceptionDialog = new ExceptionDialogModel(obj, ex.Message, _progressView);
                        exceptionDialog.Show();
                        result = exceptionDialog.CloseResult;
                    });
                    switch (result)
                    {
                        case ExceptionDialogResult.TryAgain:
                            Sign(obj);
                            return;
                        case ExceptionDialogResult.Skip:
                            return;
                        case null:
                            _cancelToken.Cancel();
                            System.Windows.Application.Current.Dispatcher.InvokeAsync(() => _progressView.Close());
                            return;
                    }
                }
            }
        }
    }
}
